"""
RestaurantAI Pro - Aplicación Principal
Análisis Financiero Inteligente para Restauración
© 2026 Roberto Martín Gutiérrez & Inés Fraile Verdugo
"""

import customtkinter as ctk
from tkinter import filedialog, messagebox, scrolledtext
import tkinter as tk
import os
import sys
import json
import math
from datetime import datetime
from typing import Dict, List, Optional

# Ajustar path para imports
if getattr(sys, 'frozen', False):
    BASE_DIR = os.path.dirname(sys.executable)
else:
    BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, BASE_DIR)

from config import *
from core.licensing import LicenseManager
from core.analysis_engine import Plato, DatosRestaurante, AnalisisFinanciero
from core.pdf_extractor import PDFExtractor
from core.chatbot import RestaurantChatbot
from core.export_reports import ReportExporter
from core.data_persistence import save_session, load_session, apply_session

try:
    from core.ai_engine import RestaurantAIEngine
    HAS_AI = True
except ImportError:
    HAS_AI = False

try:
    from core.daily_register import DailyRegister
    HAS_DAILY = True
except ImportError:
    HAS_DAILY = False

try:
    from core.updater import AutoUpdater
    HAS_UPDATER = True
except ImportError:
    HAS_UPDATER = False

try:
    from PIL import Image
    HAS_PIL = True
except ImportError:
    HAS_PIL = False

try:
    import matplotlib
    matplotlib.use('Agg')
    import matplotlib.pyplot as plt
    from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg
    import matplotlib.patches as mpatches
    HAS_MATPLOTLIB = True
except ImportError:
    HAS_MATPLOTLIB = False


# ============================================================
# TEMA Y ESTILOS
# ============================================================

ctk.set_appearance_mode("dark")
ctk.set_default_color_theme("blue")


class AppStyles:
    """Estilos centralizados."""
    FONT_TITLE = ("Segoe UI", 28, "bold")
    FONT_SUBTITLE = ("Segoe UI", 18)
    FONT_HEADING = ("Segoe UI", 16, "bold")
    FONT_BODY = ("Segoe UI", 13)
    FONT_SMALL = ("Segoe UI", 11)
    FONT_TINY = ("Segoe UI", 9)
    FONT_MONO = ("Consolas", 12)

    PADDING = 20
    CARD_CORNER = 12
    BUTTON_CORNER = 8
    BUTTON_HEIGHT = 40


# ============================================================
# COMPONENTES REUTILIZABLES
# ============================================================

class Card(ctk.CTkFrame):
    """Tarjeta con estilo consistente."""
    def __init__(self, parent, title=None, **kwargs):
        super().__init__(parent, fg_color=COLORS["card_bg"],
                         corner_radius=AppStyles.CARD_CORNER,
                         border_width=1, border_color=COLORS["card_border"],
                         **kwargs)
        if title:
            lbl = ctk.CTkLabel(self, text=title, font=AppStyles.FONT_HEADING,
                               text_color=COLORS["gold"])
            lbl.pack(anchor="w", padx=16, pady=(12, 4))


class MetricCard(ctk.CTkFrame):
    """Tarjeta de métrica individual."""
    def __init__(self, parent, label, value, unit="", color=None, **kwargs):
        super().__init__(parent, fg_color=COLORS["card_bg"],
                         corner_radius=10, border_width=1,
                         border_color=COLORS["card_border"], **kwargs)

        color = color or COLORS["text_white"]

        lbl = ctk.CTkLabel(self, text=label, font=AppStyles.FONT_SMALL,
                           text_color=COLORS["text_muted"])
        lbl.pack(anchor="w", padx=14, pady=(10, 0))

        val_text = f"{value}{unit}" if unit else str(value)
        val = ctk.CTkLabel(self, text=val_text, font=("Segoe UI", 22, "bold"),
                           text_color=color)
        val.pack(anchor="w", padx=14, pady=(2, 10))


class StyledEntry(ctk.CTkEntry):
    """Entry con estilo consistente."""
    def __init__(self, parent, placeholder="", width=200, **kwargs):
        super().__init__(parent, placeholder_text=placeholder, width=width,
                         height=38, corner_radius=8,
                         fg_color=COLORS["input_bg"],
                         border_color=COLORS["input_border"],
                         text_color=COLORS["text_white"],
                         placeholder_text_color=COLORS["text_muted"],
                         font=AppStyles.FONT_BODY, **kwargs)


class StyledButton(ctk.CTkButton):
    """Botón con estilo consistente."""
    def __init__(self, parent, text, command=None, style="primary", width=180, **kwargs):
        colors = {
            "primary": (COLORS["accent"], COLORS["accent_hover"]),
            "secondary": (COLORS["bg_light"], COLORS["card_border"]),
            "gold": (COLORS["gold"], COLORS["gold_light"]),
            "success": (COLORS["success"], "#27ae60"),
            "danger": (COLORS["danger"], "#c0392b"),
        }
        fg, hover = colors.get(style, colors["primary"])
        text_color = COLORS["text_white"] if style != "gold" else COLORS["bg_dark"]

        super().__init__(parent, text=text, command=command, width=width,
                         height=AppStyles.BUTTON_HEIGHT,
                         corner_radius=AppStyles.BUTTON_CORNER,
                         fg_color=fg, hover_color=hover,
                         text_color=text_color,
                         font=("Segoe UI", 13, "bold"), **kwargs)


# ============================================================
# APLICACIÓN PRINCIPAL
# ============================================================

class RestaurantAIApp(ctk.CTk):
    """Aplicación principal RestaurantAI Pro."""

    def __init__(self):
        super().__init__()

        # Configuración ventana
        self.title(f"{APP_NAME} v{APP_VERSION}")
        self.geometry("1280x800")
        self.minsize(1100, 700)
        self.configure(fg_color=COLORS["bg_dark"])

        # Estado
        self.license_mgr = LicenseManager()
        self.is_pro = self.license_mgr.is_pro()
        self.datos = DatosRestaurante()
        self.analisis = None
        self.pdf_extractor = PDFExtractor()
        self.chatbot = None
        self.extracted_data = {}
        self.current_screen = None
        self.ai_engine = RestaurantAIEngine() if HAS_AI else None
        self.session_file = None
        self.daily_register = DailyRegister() if HAS_DAILY else None

        # Logo
        self.logo_image = None
        self._load_logo()

        # Icono
        try:
            icon_path = os.path.join(BASE_DIR, "assets", "icon.ico")
            if os.path.exists(icon_path):
                self.iconbitmap(icon_path)
        except Exception:
            pass

        # Layout principal
        self._create_layout()
        self._show_welcome()

        # Comprobar actualizaciones al inicio (en segundo plano)
        if HAS_UPDATER and self.is_pro:
            self.after(3000, self._check_updates_background)

    def _load_logo(self):
        """Carga el logo personalizado."""
        if not HAS_PIL:
            return
        logo_paths = [
            os.path.join(BASE_DIR, "assets", "logo.png"),
            os.path.join(BASE_DIR, "logo.png"),
        ]
        for path in logo_paths:
            if os.path.exists(path):
                try:
                    self.logo_image = ctk.CTkImage(
                        light_image=Image.open(path),
                        dark_image=Image.open(path),
                        size=(180, 60)
                    )
                except Exception:
                    pass
                break

    def _create_layout(self):
        """Crea el layout principal con sidebar."""
        # Sidebar
        self.sidebar = ctk.CTkFrame(self, fg_color=COLORS["sidebar_bg"], width=240,
                                     corner_radius=0)
        self.sidebar.pack(side="left", fill="y")
        self.sidebar.pack_propagate(False)

        # Logo en sidebar
        if self.logo_image:
            logo_label = ctk.CTkLabel(self.sidebar, image=self.logo_image, text="")
            logo_label.pack(pady=(20, 5))
        else:
            title_label = ctk.CTkLabel(self.sidebar, text="🍽️ Convastro",
                                        font=("Segoe UI", 20, "bold"),
                                        text_color=COLORS["gold"])
            title_label.pack(pady=(20, 2))

        version_label = ctk.CTkLabel(self.sidebar, text=f"v{APP_VERSION}",
                                      font=AppStyles.FONT_TINY,
                                      text_color=COLORS["text_muted"])
        version_label.pack(pady=(0, 5))

        # Estado licencia
        license_text = "🔓 PRO" if self.is_pro else "🔒 DEMO"
        license_color = COLORS["success"] if self.is_pro else COLORS["warning"]
        self.license_label = ctk.CTkLabel(self.sidebar, text=license_text,
                                           font=("Segoe UI", 12, "bold"),
                                           text_color=license_color)
        self.license_label.pack(pady=(0, 15))

        # Separador
        sep = ctk.CTkFrame(self.sidebar, height=1, fg_color=COLORS["card_border"])
        sep.pack(fill="x", padx=20, pady=5)

        # Botones navegación
        self.nav_buttons = {}
        nav_items = [
            ("inicio", "🏠  Inicio", self._show_welcome),
            ("datos", "📥  Datos", self._show_data_input),
            ("revision", "🔍  Revisión", self._show_review),
            ("analisis", "📊  Análisis", self._show_analysis),
            ("ia", "🤖  IA Predictiva", self._show_ai_screen),
            ("registro", "📅  Registro Diario", self._show_daily_register),
            ("chatbot", "💬  Asistente", self._show_chatbot),
            ("guardar", "💾  Guardar/Cargar", self._show_save_load),
            ("licencia", "🔑  Licencia", self._show_license),
            ("acerca", "ℹ️  Acerca de", self._show_about),
        ]

        for key, text, cmd in nav_items:
            btn = ctk.CTkButton(
                self.sidebar, text=text, command=cmd,
                width=200, height=42, corner_radius=8,
                fg_color="transparent",
                hover_color=COLORS["sidebar_hover"],
                text_color=COLORS["text_light"],
                font=("Segoe UI", 13),
                anchor="w"
            )
            btn.pack(padx=15, pady=2)
            self.nav_buttons[key] = btn

        # Contacto al fondo
        spacer = ctk.CTkFrame(self.sidebar, fg_color="transparent")
        spacer.pack(fill="both", expand=True)

        sep2 = ctk.CTkFrame(self.sidebar, height=1, fg_color=COLORS["card_border"])
        sep2.pack(fill="x", padx=20, pady=5)

        contact = ctk.CTkLabel(self.sidebar, text=f"📧 {CONTACT_EMAIL}\n📱 {CONTACT_PHONE}",
                                font=AppStyles.FONT_TINY,
                                text_color=COLORS["text_muted"])
        contact.pack(pady=(5, 10))

        # Área principal
        self.main_area = ctk.CTkFrame(self, fg_color=COLORS["bg_dark"], corner_radius=0)
        self.main_area.pack(side="right", fill="both", expand=True)

    def _highlight_nav(self, key):
        """Resalta el botón de navegación activo."""
        for k, btn in self.nav_buttons.items():
            if k == key:
                btn.configure(fg_color=COLORS["bg_light"], text_color=COLORS["text_white"])
            else:
                btn.configure(fg_color="transparent", text_color=COLORS["text_light"])

    def _clear_main(self):
        """Limpia el área principal."""
        for widget in self.main_area.winfo_children():
            widget.destroy()

    def _create_scrollable_frame(self) -> ctk.CTkScrollableFrame:
        """Crea un frame scrollable en el área principal."""
        scroll = ctk.CTkScrollableFrame(self.main_area, fg_color=COLORS["bg_dark"],
                                         scrollbar_button_color=COLORS["card_border"],
                                         scrollbar_button_hover_color=COLORS["bg_light"])
        scroll.pack(fill="both", expand=True, padx=20, pady=20)
        return scroll

    # ============================================================
    # PANTALLA: BIENVENIDA
    # ============================================================

    def _show_welcome(self):
        """Pantalla de bienvenida."""
        self._clear_main()
        self._highlight_nav("inicio")
        self.current_screen = "inicio"

        scroll = self._create_scrollable_frame()

        # Header
        header = ctk.CTkFrame(scroll, fg_color="transparent")
        header.pack(fill="x", pady=(0, 20))

        title = ctk.CTkLabel(header, text=f"Bienvenido a {APP_NAME}",
                              font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"])
        title.pack(anchor="w")

        subtitle = ctk.CTkLabel(header, text=APP_SUBTITLE,
                                 font=AppStyles.FONT_SUBTITLE,
                                 text_color=COLORS["text_light"])
        subtitle.pack(anchor="w", pady=(4, 0))

        # Tarjetas de inicio rápido
        cards_frame = ctk.CTkFrame(scroll, fg_color="transparent")
        cards_frame.pack(fill="x", pady=10)
        cards_frame.columnconfigure((0, 1, 2), weight=1)

        # Card 1: Subir PDFs
        card1 = Card(cards_frame, title="📄 Opción A: Subir Documentos")
        card1.grid(row=0, column=0, padx=8, pady=8, sticky="nsew")
        ctk.CTkLabel(card1, text="Sube menús, facturas o resúmenes\n"
                     "de TPV en PDF y la IA extraerá\ntodos los datos automáticamente.",
                     font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                     justify="left").pack(padx=16, pady=8)
        StyledButton(card1, text="Subir PDFs →", command=lambda: self._show_data_input("pdf"),
                     style="primary", width=160).pack(padx=16, pady=(4, 16))

        # Card 2: Rellenar plantilla
        card2 = Card(cards_frame, title="✏️ Opción B: Rellenar Plantilla")
        card2.grid(row=0, column=1, padx=8, pady=8, sticky="nsew")
        ctk.CTkLabel(card2, text="Introduce los datos manualmente\nen un formulario guiado.\n"
                     "Ideal si no tienes TPV.",
                     font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                     justify="left").pack(padx=16, pady=8)
        StyledButton(card2, text="Rellenar datos →", command=lambda: self._show_data_input("manual"),
                     style="primary", width=160).pack(padx=16, pady=(4, 16))

        # Card 3: Demo / Pro
        card3 = Card(cards_frame, title="🔑 Versión Demo / Pro")
        card3.grid(row=0, column=2, padx=8, pady=8, sticky="nsew")
        if self.is_pro:
            ctk.CTkLabel(card3, text="✅ Licencia PRO activa\n\nTienes acceso completo a\n"
                         "todos los análisis y funciones.",
                         font=AppStyles.FONT_BODY, text_color=COLORS["success"],
                         justify="left").pack(padx=16, pady=8)
        else:
            ctk.CTkLabel(card3, text="La versión Demo incluye:\n"
                         "• Menu Engineering básico\n"
                         "• Hasta 10 platos\n"
                         "• Análisis a corto plazo\n\n"
                         "PRO desbloquea todo.",
                         font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                         justify="left").pack(padx=16, pady=8)
            StyledButton(card3, text="Activar PRO →", command=self._show_license,
                         style="gold", width=160).pack(padx=16, pady=(4, 16))

        # Qué incluye la app
        features_card = Card(scroll, title="📊 ¿Qué analiza Convastro?")
        features_card.pack(fill="x", pady=10)

        features_frame = ctk.CTkFrame(features_card, fg_color="transparent")
        features_frame.pack(fill="x", padx=16, pady=(4, 16))
        features_frame.columnconfigure((0, 1), weight=1)

        demo_features = [
            ("✅", "Estructura de costes (fijos, variables, directos, indirectos)"),
            ("✅", "Cálculo de márgenes y precios de venta"),
            ("✅", "Menu Engineering (Estrella, Vaca, Puzzle, Perro)"),
            ("✅", "Punto muerto / Umbral de rentabilidad"),
            ("✅", "Análisis operativo a corto plazo"),
        ]

        pro_features = [
            ("🔒" if not self.is_pro else "✅", "Análisis de demanda a medio plazo"),
            ("🔒" if not self.is_pro else "✅", "Proyección financiera a largo plazo"),
            ("🔒" if not self.is_pro else "✅", "Simulaciones What-If (precios, costes, clientes)"),
            ("🔒" if not self.is_pro else "✅", "IA Predictiva real (4 modelos Machine Learning)"),
            ("🔒" if not self.is_pro else "✅", "Registro diario con aprendizaje automático"),
            ("🔒" if not self.is_pro else "✅", "Chatbot que explica cada cálculo paso a paso"),
            ("🔒" if not self.is_pro else "✅", "Benchmarking vs sector + Health Score"),
            ("🔒" if not self.is_pro else "✅", "Exportación PDF/Excel profesional"),
            ("🔒" if not self.is_pro else "✅", "Actualizaciones remotas automáticas"),
        ]

        for i, (icon, text) in enumerate(demo_features):
            ctk.CTkLabel(features_frame, text=f"{icon} {text}",
                         font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                         anchor="w").grid(row=i, column=0, sticky="w", padx=8, pady=3)

        for i, (icon, text) in enumerate(pro_features):
            color = COLORS["text_light"] if icon == "✅" else COLORS["text_muted"]
            ctk.CTkLabel(features_frame, text=f"{icon} {text}",
                         font=AppStyles.FONT_BODY, text_color=color,
                         anchor="w").grid(row=i, column=1, sticky="w", padx=8, pady=3)

    # ============================================================
    # PANTALLA: ENTRADA DE DATOS
    # ============================================================

    def _show_data_input(self, mode=None):
        """Pantalla de entrada de datos (PDF o manual)."""
        self._clear_main()
        self._highlight_nav("datos")
        self.current_screen = "datos"

        scroll = self._create_scrollable_frame()

        title = ctk.CTkLabel(scroll, text="📥 Entrada de Datos",
                              font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"])
        title.pack(anchor="w", pady=(0, 5))

        # Tabs
        tab_frame = ctk.CTkFrame(scroll, fg_color="transparent")
        tab_frame.pack(fill="x", pady=10)

        self._data_mode = mode or "manual"

        btn_pdf = StyledButton(tab_frame, text="📄 Subir PDFs",
                               command=lambda: self._switch_data_mode("pdf", scroll),
                               style="primary" if self._data_mode == "pdf" else "secondary",
                               width=180)
        btn_pdf.pack(side="left", padx=5)

        btn_manual = StyledButton(tab_frame, text="✏️ Plantilla Manual",
                                  command=lambda: self._switch_data_mode("manual", scroll),
                                  style="primary" if self._data_mode == "manual" else "secondary",
                                  width=180)
        btn_manual.pack(side="left", padx=5)

        # Contenedor de modo
        self.data_content = ctk.CTkFrame(scroll, fg_color="transparent")
        self.data_content.pack(fill="both", expand=True, pady=10)

        if self._data_mode == "pdf":
            self._build_pdf_input(self.data_content)
        else:
            self._build_manual_input(self.data_content)

    def _switch_data_mode(self, mode, scroll):
        self._data_mode = mode
        self._show_data_input(mode)

    def _build_pdf_input(self, parent):
        """Construye la interfaz de carga de PDFs."""
        info = ctk.CTkLabel(parent,
                            text="Sube documentos PDF de tu restaurante. La IA extraerá los datos automáticamente.\n"
                            "Puedes subir: Menús/Cartas, Facturas de proveedores, Resúmenes de TPV.",
                            font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                            justify="left")
        info.pack(anchor="w", pady=(0, 15))

        # Área de subida
        upload_frame = Card(parent, title="Documentos subidos")
        upload_frame.pack(fill="x", pady=5)

        self.pdf_listbox_frame = ctk.CTkFrame(upload_frame, fg_color="transparent")
        self.pdf_listbox_frame.pack(fill="x", padx=16, pady=8)

        self.uploaded_files = []

        btn_frame = ctk.CTkFrame(upload_frame, fg_color="transparent")
        btn_frame.pack(fill="x", padx=16, pady=(0, 12))

        StyledButton(btn_frame, text="📁 Añadir PDF", command=self._add_pdf,
                     style="primary", width=150).pack(side="left", padx=5)
        StyledButton(btn_frame, text="🗑️ Limpiar", command=self._clear_pdfs,
                     style="danger", width=120).pack(side="left", padx=5)
        StyledButton(btn_frame, text="🔍 Extraer Datos →",
                     command=self._extract_pdfs,
                     style="gold", width=180).pack(side="right", padx=5)

        # Datos adicionales necesarios
        extra_card = Card(parent, title="Datos complementarios (necesarios)")
        extra_card.pack(fill="x", pady=10)

        extra_frame = ctk.CTkFrame(extra_card, fg_color="transparent")
        extra_frame.pack(fill="x", padx=16, pady=(4, 16))
        extra_frame.columnconfigure((0, 1, 2, 3), weight=1)

        self.pdf_extra_entries = {}
        extra_fields = [
            ("nombre", "Nombre del restaurante", 0, 0),
            ("tipo", "Tipo (tradicional, fast food...)", 0, 1),
            ("asientos", "Nº asientos", 0, 2),
            ("dias_mes", "Días apertura/mes", 0, 3),
            ("servicios_dia", "Servicios/día", 1, 0),
            ("clientes_mes", "Clientes/mes", 1, 1),
            ("alquiler", "Alquiler mensual (€)", 1, 2),
            ("salarios", "Salarios totales (€)", 1, 3),
            ("seg_social", "Seg. Social (€)", 2, 0),
            ("seguros", "Seguros (€)", 2, 1),
            ("suministros", "Suministros fijos (€)", 2, 2),
            ("amortizaciones", "Amortizaciones (€)", 2, 3),
        ]

        for key, label, row, col in extra_fields:
            frame = ctk.CTkFrame(extra_frame, fg_color="transparent")
            frame.grid(row=row, column=col, padx=5, pady=5, sticky="ew")
            ctk.CTkLabel(frame, text=label, font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_muted"]).pack(anchor="w")
            entry = StyledEntry(frame, placeholder=label, width=200)
            entry.pack(fill="x")
            self.pdf_extra_entries[key] = entry

    def _build_manual_input(self, parent):
        """Construye la plantilla de entrada manual completa."""
        info = ctk.CTkLabel(parent,
                            text="Rellena los datos de tu restaurante. Todos los campos son importantes "
                            "para un análisis preciso.",
                            font=AppStyles.FONT_BODY, text_color=COLORS["text_light"])
        info.pack(anchor="w", pady=(0, 15))

        self.manual_entries = {}

        # SECCIÓN 1: Datos generales
        card1 = Card(parent, title="🏪 Datos Generales")
        card1.pack(fill="x", pady=5)
        f1 = ctk.CTkFrame(card1, fg_color="transparent")
        f1.pack(fill="x", padx=16, pady=(4, 16))
        f1.columnconfigure((0, 1, 2, 3), weight=1)

        general_fields = [
            ("nombre", "Nombre del restaurante", 0, 0),
            ("tipo", "Tipo (tradicional, bufé, fast food...)", 0, 1),
            ("asientos", "Nº asientos", 0, 2),
            ("personal", "Nº personal total", 0, 3),
            ("dias_mes", "Días apertura/mes", 1, 0),
            ("servicios_dia", "Servicios/día (ej: 2)", 1, 1),
            ("clientes_mes", "Clientes totales/mes", 1, 2),
        ]
        self._build_fields(f1, general_fields)

        # SECCIÓN 2: Costes fijos mensuales
        card2 = Card(parent, title="🏠 Costes Fijos Mensuales (€)")
        card2.pack(fill="x", pady=5)
        f2 = ctk.CTkFrame(card2, fg_color="transparent")
        f2.pack(fill="x", padx=16, pady=(4, 16))
        f2.columnconfigure((0, 1, 2, 3), weight=1)

        fijos_fields = [
            ("alquiler", "Alquiler", 0, 0),
            ("salarios_fijos", "Salarios fijos", 0, 1),
            ("seg_social", "Seguridad Social", 0, 2),
            ("seguros", "Seguros", 0, 3),
            ("suministros_fijos", "Suministros fijos", 1, 0),
            ("amortizaciones", "Amortizaciones", 1, 1),
            ("otros_fijos", "Otros fijos", 1, 2),
        ]
        self._build_fields(f2, fijos_fields)

        # SECCIÓN 3: Costes variables
        card3 = Card(parent, title="📦 Costes Variables Mensuales (€)")
        card3.pack(fill="x", pady=5)
        f3 = ctk.CTkFrame(card3, fg_color="transparent")
        f3.pack(fill="x", padx=16, pady=(4, 16))
        f3.columnconfigure((0, 1, 2, 3), weight=1)

        var_fields = [
            ("coste_mp_total", "Coste materia prima total", 0, 0),
            ("salarios_variables", "Salarios variables/extras", 0, 1),
            ("suministros_var", "Suministros variables", 0, 2),
            ("otros_variables", "Otros variables", 0, 3),
        ]
        self._build_fields(f3, var_fields)

        # SECCIÓN 4: Stock
        card4 = Card(parent, title="📋 Datos de Stock (€)")
        card4.pack(fill="x", pady=5)
        f4 = ctk.CTkFrame(card4, fg_color="transparent")
        f4.pack(fill="x", padx=16, pady=(4, 16))
        f4.columnconfigure((0, 1, 2), weight=1)

        stock_fields = [
            ("stock_inicial", "Stock inicial mes", 0, 0),
            ("stock_final", "Stock final mes", 0, 1),
            ("compras_periodo", "Compras del periodo", 0, 2),
        ]
        self._build_fields(f4, stock_fields)

        # SECCIÓN 5: Platos de la carta
        card5 = Card(parent, title="🍽️ Carta / Menú (platos)")
        card5.pack(fill="x", pady=5)

        platos_info = ctk.CTkLabel(card5,
                                    text="Añade cada plato con su precio de venta (sin IVA), "
                                    "coste de materia prima y unidades vendidas al mes.",
                                    font=AppStyles.FONT_SMALL,
                                    text_color=COLORS["text_muted"])
        platos_info.pack(padx=16, anchor="w")

        # Header de tabla de platos
        header_frame = ctk.CTkFrame(card5, fg_color=COLORS["bg_light"],
                                     corner_radius=6)
        header_frame.pack(fill="x", padx=16, pady=(8, 0))
        headers = ["Nombre del plato", "Categoría", "PV sin IVA (€)",
                   "Coste MP (€)", "Uds vendidas/mes", ""]
        widths = [220, 120, 120, 120, 130, 60]
        for i, (h, w) in enumerate(zip(headers, widths)):
            ctk.CTkLabel(header_frame, text=h, font=("Segoe UI", 11, "bold"),
                         text_color=COLORS["gold"], width=w).pack(side="left", padx=4, pady=6)

        # Lista de platos
        self.platos_frame = ctk.CTkFrame(card5, fg_color="transparent")
        self.platos_frame.pack(fill="x", padx=16, pady=4)
        self.platos_rows = []

        # Añadir 3 filas iniciales
        for _ in range(3):
            self._add_plato_row()

        btn_frame5 = ctk.CTkFrame(card5, fg_color="transparent")
        btn_frame5.pack(fill="x", padx=16, pady=(4, 16))
        StyledButton(btn_frame5, text="+ Añadir plato", command=self._add_plato_row,
                     style="secondary", width=140).pack(side="left", padx=5)

        limit_text = f"(Máx. {DEMO_LIMITS['max_platos']} en demo)" if not self.is_pro else ""
        if limit_text:
            ctk.CTkLabel(btn_frame5, text=limit_text, font=AppStyles.FONT_SMALL,
                         text_color=COLORS["warning"]).pack(side="left", padx=10)

        # Botón guardar y continuar
        action_frame = ctk.CTkFrame(parent, fg_color="transparent")
        action_frame.pack(fill="x", pady=15)

        StyledButton(action_frame, text="💾 Guardar y Revisar →",
                     command=self._save_manual_data,
                     style="gold", width=220).pack(side="right", padx=5)

        # Pre-rellenar si ya hay datos cargados (ej: volver de revisión o PDF)
        self._prefill_manual_entries()

    def _build_fields(self, parent, fields):
        """Construye campos de entrada en un grid."""
        for key, label, row, col in fields:
            frame = ctk.CTkFrame(parent, fg_color="transparent")
            frame.grid(row=row, column=col, padx=5, pady=5, sticky="ew")
            ctk.CTkLabel(frame, text=label, font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_muted"]).pack(anchor="w")
            entry = StyledEntry(frame, placeholder="0", width=200)
            entry.pack(fill="x")
            self.manual_entries[key] = entry

    def _add_plato_row(self):
        """Añade una fila para un nuevo plato."""
        max_platos = PRO_FEATURES["max_platos"] if self.is_pro else DEMO_LIMITS["max_platos"]
        if len(self.platos_rows) >= max_platos:
            messagebox.showwarning("Límite Demo",
                                   f"La versión Demo permite máximo {DEMO_LIMITS['max_platos']} platos.\n"
                                   "Activa la versión PRO para platos ilimitados.")
            return

        row_frame = ctk.CTkFrame(self.platos_frame, fg_color="transparent")
        row_frame.pack(fill="x", pady=2)

        entries = {}
        entries["nombre"] = StyledEntry(row_frame, placeholder="Ej: Solomillo", width=220)
        entries["nombre"].pack(side="left", padx=2)

        entries["categoria"] = ctk.CTkComboBox(
            row_frame, values=["entrante", "principal", "postre", "bebida"],
            width=120, height=38, fg_color=COLORS["input_bg"],
            border_color=COLORS["input_border"], button_color=COLORS["bg_light"],
            dropdown_fg_color=COLORS["card_bg"], font=AppStyles.FONT_BODY,
            text_color=COLORS["text_white"]
        )
        entries["categoria"].set("principal")
        entries["categoria"].pack(side="left", padx=2)

        entries["precio"] = StyledEntry(row_frame, placeholder="0.00", width=120)
        entries["precio"].pack(side="left", padx=2)

        entries["coste_mp"] = StyledEntry(row_frame, placeholder="0.00", width=120)
        entries["coste_mp"].pack(side="left", padx=2)

        entries["unidades"] = StyledEntry(row_frame, placeholder="0", width=130)
        entries["unidades"].pack(side="left", padx=2)

        def remove_row(rf=row_frame, idx=len(self.platos_rows)):
            rf.destroy()
            self.platos_rows = [r for r in self.platos_rows if r["frame"] != rf]

        btn_del = ctk.CTkButton(row_frame, text="✕", width=40, height=38,
                                 fg_color=COLORS["danger"], hover_color="#c0392b",
                                 corner_radius=6, command=remove_row,
                                 font=("Segoe UI", 14, "bold"))
        btn_del.pack(side="left", padx=2)

        self.platos_rows.append({"frame": row_frame, "entries": entries})

    # ============================================================
    # PRE-RELLENADO DE DATOS (al volver atrás)
    # ============================================================

    def _prefill_manual_entries(self):
        """Pre-rellena campos si ya hay datos en self.datos."""
        if not self.datos:
            return
        prefill_map = {
            "nombre": self.datos.nombre_restaurante,
            "tipo": self.datos.tipo_restaurante,
            "asientos": self.datos.asientos,
            "personal": self.datos.personal_total,
            "dias_mes": self.datos.dias_apertura_mes,
            "servicios_dia": self.datos.servicios_por_dia,
            "clientes_mes": self.datos.clientes_mes,
            "alquiler": self.datos.alquiler,
            "salarios_fijos": self.datos.salarios_fijos,
            "seg_social": self.datos.seguridad_social,
            "seguros": self.datos.seguros,
            "suministros_fijos": self.datos.suministros_fijos,
            "amortizaciones": self.datos.amortizaciones,
            "otros_fijos": self.datos.otros_fijos,
            "coste_mp_total": self.datos.coste_mp_total,
            "salarios_variables": self.datos.salarios_variables,
            "suministros_var": self.datos.suministros_variables,
            "otros_variables": self.datos.otros_variables,
            "stock_inicial": self.datos.stock_inicial,
            "stock_final": self.datos.stock_final,
            "compras_periodo": self.datos.compras_periodo,
        }
        for key, val in prefill_map.items():
            entry = self.manual_entries.get(key)
            if entry and val:
                try:
                    entry.delete(0, "end")
                    entry.insert(0, str(val))
                except Exception:
                    pass

        # Pre-rellenar platos existentes
        if self.datos.platos and len(self.platos_rows) < len(self.datos.platos):
            # Limpiar filas vacías
            for row in self.platos_rows:
                row["frame"].destroy()
            self.platos_rows = []
            # Crear filas con datos
            for plato in self.datos.platos:
                self._add_plato_row()
                row = self.platos_rows[-1]
                row["entries"]["nombre"].insert(0, plato.nombre)
                row["entries"]["categoria"].set(plato.categoria)
                row["entries"]["precio"].insert(0, f"{plato.precio_venta:.2f}")
                row["entries"]["coste_mp"].insert(0, f"{plato.coste_mp:.2f}")
                row["entries"]["unidades"].insert(0, str(plato.unidades_vendidas))

    # ============================================================
    # COMPROBACIÓN DE ACTUALIZACIONES
    # ============================================================

    def _check_updates_background(self):
        """Comprueba actualizaciones en segundo plano."""
        try:
            updater = AutoUpdater()
            info = updater.check_update()
            if info and info.get("update_available"):
                self._show_update_banner(info)
        except Exception:
            pass

    def _show_update_banner(self, info):
        """Muestra banner de actualización disponible."""
        banner = ctk.CTkFrame(self.main_area, fg_color=COLORS["bg_light"],
                               corner_radius=10, height=50)
        banner.pack(fill="x", padx=20, pady=(5, 0), before=self.main_area.winfo_children()[0]
                    if self.main_area.winfo_children() else None)

        ctk.CTkLabel(banner, text=f"🔄 Nueva versión {info.get('version', '')} disponible — {info.get('changelog', '')}",
                     font=("Segoe UI", 12), text_color=COLORS["gold"]).pack(side="left", padx=16, pady=10)

        def do_update():
            import webbrowser
            webbrowser.open(info.get("download_url", "https://convastro.es/#download"))

        StyledButton(banner, text="Actualizar", command=do_update,
                     style="gold", width=100).pack(side="right", padx=16, pady=8)

    # ============================================================
    # PANTALLA: REGISTRO DIARIO (PRO)
    # ============================================================

    def _show_daily_register(self):
        """Pantalla de registro diario con historial e IA."""
        self._clear_main()
        self._highlight_nav("registro")

        if not self.is_pro:
            frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
            frame.pack(fill="both", expand=True, padx=40, pady=40)
            ctk.CTkLabel(frame, text="📅 Registro Diario", font=AppStyles.FONT_TITLE,
                         text_color=COLORS["text_white"]).pack(pady=(40, 10))
            ctk.CTkLabel(frame, text="🔒 Exclusivo versión PRO", font=AppStyles.FONT_SUBTITLE,
                         text_color=COLORS["gold"]).pack(pady=10)
            ctk.CTkLabel(frame, text="El registro diario te permite:\n\n"
                         "• Registrar clientes, ingresos y costes cada día\n"
                         "• Anotar platos vendidos por servicio\n"
                         "• La IA aprende de tus datos reales con el tiempo\n"
                         "• Predicciones personalizadas basadas en tu historial\n"
                         "• Detectar tendencias y patrones automáticamente\n"
                         "• Saber qué día de la semana es más rentable\n"
                         "• Recomendaciones que mejoran cuantos más datos tengas",
                         font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                         justify="left").pack(pady=15)
            StyledButton(frame, text="🔑 Activar PRO →", command=self._show_license,
                         style="gold", width=200).pack(pady=15)
            return

        if not self.daily_register:
            frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
            frame.pack(fill="both", expand=True, padx=40, pady=40)
            ctk.CTkLabel(frame, text="Módulo de registro no disponible.",
                         font=AppStyles.FONT_SUBTITLE, text_color=COLORS["warning"]).pack(pady=40)
            return

        scroll = self._create_scrollable_frame()

        ctk.CTkLabel(scroll, text="📅 Registro Diario — Aprende de tu negocio",
                     font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"]).pack(anchor="w", pady=(0, 15))

        # ====== FORMULARIO DE REGISTRO ======
        form_card = Card(scroll, title="➕ Registrar día")
        form_card.pack(fill="x", pady=5)

        form_inner = ctk.CTkFrame(form_card, fg_color="transparent")
        form_inner.pack(fill="x", padx=16, pady=10)

        self.daily_entries = {}

        # Fila 1: Fecha + Día semana
        row1 = ctk.CTkFrame(form_inner, fg_color="transparent")
        row1.pack(fill="x", pady=3)
        ctk.CTkLabel(row1, text="Fecha:", font=AppStyles.FONT_SMALL,
                     text_color=COLORS["text_light"], width=100).pack(side="left")
        self.daily_entries["fecha"] = ctk.CTkEntry(row1, width=130, fg_color=COLORS["input_bg"],
                                                    border_color=COLORS["input_border"])
        self.daily_entries["fecha"].insert(0, datetime.now().strftime("%Y-%m-%d"))
        self.daily_entries["fecha"].pack(side="left", padx=5)

        dias = ["lunes", "martes", "miércoles", "jueves", "viernes", "sábado", "domingo"]
        ctk.CTkLabel(row1, text="Día:", font=AppStyles.FONT_SMALL,
                     text_color=COLORS["text_light"], width=50).pack(side="left", padx=(15, 0))
        self.daily_entries["dia_semana"] = ctk.CTkComboBox(row1, values=dias, width=130,
                                                            fg_color=COLORS["input_bg"],
                                                            border_color=COLORS["input_border"])
        # Auto-detectar día de la semana
        dia_idx = datetime.now().weekday()
        self.daily_entries["dia_semana"].set(dias[dia_idx])
        self.daily_entries["dia_semana"].pack(side="left", padx=5)

        # Fila 2: Clientes
        row2 = ctk.CTkFrame(form_inner, fg_color="transparent")
        row2.pack(fill="x", pady=3)
        for label, key, width in [("Clientes comida:", "clientes_comida", 80),
                                   ("Clientes cena:", "clientes_cena", 80)]:
            ctk.CTkLabel(row2, text=label, font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_light"], width=110).pack(side="left")
            self.daily_entries[key] = ctk.CTkEntry(row2, width=width, fg_color=COLORS["input_bg"],
                                                    border_color=COLORS["input_border"])
            self.daily_entries[key].pack(side="left", padx=5)

        # Fila 3: Ingresos y costes
        row3 = ctk.CTkFrame(form_inner, fg_color="transparent")
        row3.pack(fill="x", pady=3)
        for label, key in [("Ingresos del día (€):", "ingresos"),
                           ("Coste MP (€):", "costes_mp"),
                           ("Coste personal (€):", "costes_personal")]:
            ctk.CTkLabel(row3, text=label, font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_light"], width=140).pack(side="left")
            self.daily_entries[key] = ctk.CTkEntry(row3, width=90, fg_color=COLORS["input_bg"],
                                                    border_color=COLORS["input_border"])
            self.daily_entries[key].pack(side="left", padx=3)

        # Fila 4: Clima + evento + notas
        row4 = ctk.CTkFrame(form_inner, fg_color="transparent")
        row4.pack(fill="x", pady=3)
        ctk.CTkLabel(row4, text="Clima:", font=AppStyles.FONT_SMALL,
                     text_color=COLORS["text_light"], width=50).pack(side="left")
        self.daily_entries["clima"] = ctk.CTkComboBox(row4, width=110,
            values=["soleado", "nublado", "lluvioso", "frío", "caluroso"],
            fg_color=COLORS["input_bg"], border_color=COLORS["input_border"])
        self.daily_entries["clima"].pack(side="left", padx=5)

        ctk.CTkLabel(row4, text="Notas:", font=AppStyles.FONT_SMALL,
                     text_color=COLORS["text_light"], width=50).pack(side="left", padx=(10, 0))
        self.daily_entries["notas"] = ctk.CTkEntry(row4, width=300, fg_color=COLORS["input_bg"],
                                                    border_color=COLORS["input_border"],
                                                    placeholder_text="Evento especial, observaciones...")
        self.daily_entries["notas"].pack(side="left", padx=5)

        # Botón guardar
        btn_row = ctk.CTkFrame(form_inner, fg_color="transparent")
        btn_row.pack(fill="x", pady=(10, 5))
        StyledButton(btn_row, text="💾 Guardar registro", command=self._save_daily_entry,
                     style="gold", width=180).pack(side="left")

        self.daily_status_label = ctk.CTkLabel(btn_row, text="", font=AppStyles.FONT_SMALL,
                                                text_color=COLORS["success"])
        self.daily_status_label.pack(side="left", padx=15)

        # ====== RESUMEN DEL MES ======
        summary = self.daily_register.get_summary(30)
        if "error" not in summary:
            sum_card = Card(scroll, title=f"📊 Resumen últimos 30 días ({summary['dias_registrados']} días registrados)")
            sum_card.pack(fill="x", pady=5)

            metrics_frame = ctk.CTkFrame(sum_card, fg_color="transparent")
            metrics_frame.pack(fill="x", padx=16, pady=8)
            metrics_frame.columnconfigure((0, 1, 2, 3), weight=1)

            metrics = [
                ("Clientes totales", f"{summary['total_clientes']:,}"),
                ("Media clientes/día", f"{summary['media_clientes_dia']:.0f}"),
                ("Ingresos totales", f"{summary['total_ingresos']:,.0f}€"),
                ("Media ingresos/día", f"{summary['media_ingresos_dia']:,.0f}€"),
                ("Beneficio total", f"{summary['total_beneficio']:,.0f}€"),
                ("Ticket medio", f"{summary['ticket_medio']:.2f}€"),
                ("% Materia prima", f"{summary['pct_mp']:.1f}%"),
                ("Beneficio/día", f"{summary['media_beneficio_dia']:,.0f}€"),
            ]
            for i, (label, value) in enumerate(metrics):
                col = i % 4
                row = i // 4
                cell = ctk.CTkFrame(metrics_frame, fg_color=COLORS["bg_dark"], corner_radius=8)
                cell.grid(row=row, column=col, padx=4, pady=4, sticky="nsew")
                ctk.CTkLabel(cell, text=value, font=("Segoe UI", 16, "bold"),
                             text_color=COLORS["gold"]).pack(padx=10, pady=(8, 2))
                ctk.CTkLabel(cell, text=label, font=AppStyles.FONT_TINY,
                             text_color=COLORS["text_muted"]).pack(padx=10, pady=(0, 8))

            # Mejor/peor día
            if summary.get("mejor_dia"):
                info_row = ctk.CTkFrame(sum_card, fg_color="transparent")
                info_row.pack(fill="x", padx=16, pady=4)
                ctk.CTkLabel(info_row,
                    text=f"📈 Mejor día: {summary['mejor_dia']['fecha']} ({summary['mejor_dia']['ingresos']:,.0f}€) "
                         f"  |  📉 Peor día: {summary['peor_dia']['fecha']} ({summary['peor_dia']['ingresos']:,.0f}€)",
                    font=AppStyles.FONT_SMALL, text_color=COLORS["text_light"]).pack(anchor="w")

            # Top platos
            if summary.get("top_platos"):
                ctk.CTkLabel(sum_card, text="").pack()  # spacer
                top_frame = ctk.CTkFrame(sum_card, fg_color="transparent")
                top_frame.pack(fill="x", padx=16, pady=(0, 12))
                ctk.CTkLabel(top_frame, text="🏆 Platos más vendidos:",
                             font=("Segoe UI", 12, "bold"), text_color=COLORS["text_white"]).pack(anchor="w")
                for nombre, data in summary["top_platos"][:5]:
                    ctk.CTkLabel(top_frame,
                        text=f"  {nombre}: {data['cantidad']} uds ({data['ingresos']:,.0f}€)",
                        font=AppStyles.FONT_SMALL, text_color=COLORS["text_light"]).pack(anchor="w")

        # ====== TENDENCIAS IA ======
        trends = self.daily_register.ai_trends(30)
        if "error" not in trends:
            ai_card = Card(scroll, title="🤖 Análisis IA del historial")
            ai_card.pack(fill="x", pady=5)

            ai_inner = ctk.CTkFrame(ai_card, fg_color="transparent")
            ai_inner.pack(fill="x", padx=16, pady=10)

            trend_items = [
                ("Tendencia ingresos", trends["tendencia_ingresos"],
                 COLORS["success"] if "Subiendo" in trends["tendencia_ingresos"] else COLORS["danger"] if "Bajando" in trends["tendencia_ingresos"] else COLORS["text_light"]),
                ("Cambio diario", f"{trends['cambio_diario_ingresos']:+.0f}€/día",
                 COLORS["success"] if trends["cambio_diario_ingresos"] > 0 else COLORS["danger"]),
                ("Tendencia clientes", trends["tendencia_clientes"],
                 COLORS["success"] if "Subiendo" in trends["tendencia_clientes"] else COLORS["danger"] if "Bajando" in trends["tendencia_clientes"] else COLORS["text_light"]),
                ("Día más rentable", trends.get("dia_mas_rentable", "—"), COLORS["gold"]),
            ]
            ai_inner.columnconfigure((0, 1, 2, 3), weight=1)
            for i, (label, value, color) in enumerate(trend_items):
                cell = ctk.CTkFrame(ai_inner, fg_color=COLORS["bg_dark"], corner_radius=8)
                cell.grid(row=0, column=i, padx=4, pady=4, sticky="nsew")
                ctk.CTkLabel(cell, text=str(value), font=("Segoe UI", 14, "bold"),
                             text_color=color).pack(padx=10, pady=(8, 2))
                ctk.CTkLabel(cell, text=label, font=AppStyles.FONT_TINY,
                             text_color=COLORS["text_muted"]).pack(padx=10, pady=(0, 8))

            # Predicción 7 días
            pred = trends.get("prediccion_7_dias", {})
            if pred:
                ctk.CTkLabel(ai_card,
                    text=f"🔮 Predicción próximos 7 días: {pred.get('ingresos_total', 0):,.0f}€ ingresos "
                         f"({pred.get('clientes_total', 0)} clientes estimados)",
                    font=AppStyles.FONT_BODY, text_color=COLORS["blue"]).pack(padx=16, pady=(4, 4))

        # ====== RECOMENDACIONES IA ======
        recs = self.daily_register.ai_recommendations()
        if recs:
            rec_card = Card(scroll, title="💡 Recomendaciones personalizadas (basadas en tu historial)")
            rec_card.pack(fill="x", pady=5)
            for r in recs:
                ctk.CTkLabel(rec_card, text=r, font=AppStyles.FONT_BODY,
                             text_color=COLORS["text_light"], wraplength=700,
                             justify="left").pack(padx=16, pady=3, anchor="w")
            ctk.CTkLabel(rec_card, text="").pack()  # spacer

        # ====== HISTORIAL ======
        recent = self.daily_register.get_entries(14)
        if recent:
            hist_card = Card(scroll, title=f"📋 Últimos {len(recent)} registros")
            hist_card.pack(fill="x", pady=5)

            # Header
            hdr = ctk.CTkFrame(hist_card, fg_color=COLORS["bg_light"], corner_radius=6)
            hdr.pack(fill="x", padx=16, pady=(8, 2))
            for txt, w in [("Fecha", 100), ("Día", 80), ("Clientes", 70), ("Ingresos", 90),
                           ("Coste MP", 80), ("Beneficio", 90), ("Clima", 80)]:
                ctk.CTkLabel(hdr, text=txt, font=("Segoe UI", 10, "bold"),
                             text_color=COLORS["gold"], width=w).pack(side="left", padx=2, pady=5)

            for entry in reversed(recent[-14:]):
                row = ctk.CTkFrame(hist_card, fg_color="transparent")
                row.pack(fill="x", padx=16, pady=1)
                benef = entry.get("beneficio", 0)
                vals = [
                    (entry.get("fecha", ""), 100, COLORS["text_white"]),
                    (entry.get("dia_semana", ""), 80, COLORS["text_light"]),
                    (str(entry.get("clientes_total", 0)), 70, COLORS["text_light"]),
                    (f"{entry.get('ingresos', 0):,.0f}€", 90, COLORS["gold"]),
                    (f"{entry.get('costes_mp', 0):,.0f}€", 80, COLORS["text_light"]),
                    (f"{benef:,.0f}€", 90, COLORS["success"] if benef > 0 else COLORS["danger"]),
                    (entry.get("clima", ""), 80, COLORS["text_muted"]),
                ]
                for text, w, color in vals:
                    ctk.CTkLabel(row, text=text, font=AppStyles.FONT_SMALL,
                                 text_color=color, width=w).pack(side="left", padx=2)
            ctk.CTkLabel(hist_card, text="").pack(pady=4)

    def _save_daily_entry(self):
        """Guarda un registro diario."""
        try:
            fecha = self.daily_entries["fecha"].get().strip()
            datos = {
                "clientes_comida": int(self.daily_entries["clientes_comida"].get() or 0),
                "clientes_cena": int(self.daily_entries["clientes_cena"].get() or 0),
                "ingresos": float(self.daily_entries["ingresos"].get() or 0),
                "costes_mp": float(self.daily_entries["costes_mp"].get() or 0),
                "costes_personal": float(self.daily_entries["costes_personal"].get() or 0),
                "dia_semana": self.daily_entries["dia_semana"].get(),
                "clima": self.daily_entries["clima"].get(),
                "notas": self.daily_entries["notas"].get(),
            }

            if not fecha:
                self.daily_status_label.configure(text="⚠️ Introduce la fecha", text_color=COLORS["warning"])
                return

            if datos["ingresos"] == 0 and datos["clientes_comida"] == 0 and datos["clientes_cena"] == 0:
                self.daily_status_label.configure(text="⚠️ Introduce al menos clientes o ingresos",
                                                   text_color=COLORS["warning"])
                return

            self.daily_register.add_entry(fecha, datos)
            self.daily_status_label.configure(
                text=f"✅ Guardado: {fecha} ({datos['clientes_comida']+datos['clientes_cena']} clientes, {datos['ingresos']:.0f}€)",
                text_color=COLORS["success"]
            )

            # Refrescar pantalla tras 1.5s para mostrar datos actualizados
            self.after(1500, self._show_daily_register)

        except ValueError as e:
            self.daily_status_label.configure(text=f"⚠️ Valor inválido: {e}", text_color=COLORS["danger"])
        except Exception as e:
            self.daily_status_label.configure(text=f"❌ Error: {e}", text_color=COLORS["danger"])

    # ============================================================
    # PANTALLA: GUARDAR / CARGAR
    # ============================================================

    def _show_save_load(self):
        """Pantalla de guardar y cargar sesiones."""
        self._clear_main()
        self._highlight_nav("guardar")

        frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=40, pady=30)

        ctk.CTkLabel(frame, text="💾 Guardar / Cargar Análisis",
                     font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"]).pack(pady=(0, 20))

        # Guardar
        save_card = Card(frame, title="Guardar sesión actual")
        save_card.pack(fill="x", pady=10)

        has_data = bool(self.datos.platos)
        status = f"✅ {len(self.datos.platos)} platos cargados — {self.datos.nombre_restaurante}" if has_data else "⚠️ No hay datos cargados"
        ctk.CTkLabel(save_card, text=status, font=AppStyles.FONT_BODY,
                     text_color=COLORS["success"] if has_data else COLORS["warning"]).pack(padx=16, pady=8)

        StyledButton(save_card, text="💾 Guardar como...", command=self._do_save,
                     style="gold" if has_data else "secondary", width=200).pack(padx=16, pady=(0, 16))

        # Cargar
        load_card = Card(frame, title="Cargar sesión guardada")
        load_card.pack(fill="x", pady=10)

        ctk.CTkLabel(load_card, text="Carga un archivo .json con datos de un análisis anterior.",
                     font=AppStyles.FONT_BODY, text_color=COLORS["text_light"]).pack(padx=16, pady=8)

        StyledButton(load_card, text="📂 Abrir archivo...", command=self._do_load,
                     style="primary", width=200).pack(padx=16, pady=(0, 16))

        # Auto-guardado info
        info_card = Card(frame, title="ℹ️ Información")
        info_card.pack(fill="x", pady=10)
        ctk.CTkLabel(info_card,
                     text="Los archivos .json guardan todos los datos del restaurante, platos y costes.\n"
                     "Puedes compartirlos, hacer backup, o cargarlos en otro equipo con RestaurantAI.",
                     font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                     justify="left").pack(padx=16, pady=12)

    def _do_save(self):
        if not self.datos.platos:
            messagebox.showwarning("Sin datos", "No hay datos para guardar.")
            return
        filepath = filedialog.asksaveasfilename(
            title="Guardar sesión", defaultextension=".json",
            filetypes=[("JSON", "*.json")],
            initialfile=f"{self.datos.nombre_restaurante or 'sesion'}_{datetime.now().strftime('%Y%m%d')}.json"
        )
        if filepath:
            if save_session(self.datos, filepath):
                self.session_file = filepath
                messagebox.showinfo("Guardado", f"Sesión guardada en:\n{filepath}")
            else:
                messagebox.showerror("Error", "No se pudo guardar.")

    def _do_load(self):
        filepath = filedialog.askopenfilename(
            title="Cargar sesión", filetypes=[("JSON", "*.json"), ("Todos", "*.*")]
        )
        if filepath:
            session = load_session(filepath)
            if session:
                apply_session(self.datos, session, Plato)
                self.session_file = filepath
                self.analisis = AnalisisFinanciero(self.datos)
                if self.ai_engine:
                    self.ai_engine.datos = self.datos
                messagebox.showinfo("Cargado",
                    f"Sesión cargada: {self.datos.nombre_restaurante}\n"
                    f"{len(self.datos.platos)} platos recuperados.")
                self._show_review()
            else:
                messagebox.showerror("Error", "No se pudo cargar el archivo.")

    # ============================================================
    # PANTALLA: IA PREDICTIVA (PRO)
    # ============================================================

    def _show_ai_screen(self):
        """Pantalla de IA predictiva."""
        self._clear_main()
        self._highlight_nav("ia")

        if not self.is_pro:
            frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
            frame.pack(fill="both", expand=True, padx=40, pady=40)
            ctk.CTkLabel(frame, text="🤖 IA Predictiva", font=AppStyles.FONT_TITLE,
                         text_color=COLORS["text_white"]).pack(pady=(40, 10))
            ctk.CTkLabel(frame, text="🔒 Exclusivo versión PRO", font=AppStyles.FONT_SUBTITLE,
                         text_color=COLORS["gold"]).pack(pady=10)
            ctk.CTkLabel(frame, text="La IA predictiva incluye:\n\n"
                         "• Predicción de ingresos con Machine Learning (Gradient Boosting)\n"
                         "• Forecasting de demanda (Random Forest)\n"
                         "• Optimización de precios con análisis de elasticidad\n"
                         "• Clustering inteligente de platos\n"
                         "• Health Score financiero (0-100)\n"
                         "• Simulaciones What-If con modelo de elasticidad\n"
                         "• Benchmarking vs sector por tipo de restaurante",
                         font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                         justify="left").pack(pady=15)
            StyledButton(frame, text="🔑 Activar PRO →", command=self._show_license,
                         style="gold", width=200).pack(pady=15)
            return

        if not self.analisis:
            frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
            frame.pack(fill="both", expand=True, padx=40, pady=40)
            ctk.CTkLabel(frame, text="Primero carga datos y ejecuta el análisis.",
                         font=AppStyles.FONT_SUBTITLE, text_color=COLORS["warning"]).pack(pady=40)
            return

        if not self.ai_engine or not HAS_AI:
            frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
            frame.pack(fill="both", expand=True, padx=40, pady=40)
            ctk.CTkLabel(frame, text="scikit-learn no está instalado.\npip install scikit-learn",
                         font=AppStyles.FONT_SUBTITLE, text_color=COLORS["warning"]).pack(pady=40)
            return

        scroll = self._create_scrollable_frame()

        ctk.CTkLabel(scroll, text="🤖 IA Predictiva — Análisis Avanzado con Machine Learning",
                     font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"]).pack(anchor="w", pady=(0, 10))

        # Entrenar modelos
        self.ai_engine.datos = self.datos
        train_result = self.ai_engine.train_models()

        if not train_result.get("success"):
            ctk.CTkLabel(scroll, text=f"Error entrenando modelos: {train_result.get('error','')}",
                         font=AppStyles.FONT_BODY, text_color=COLORS["danger"]).pack(pady=10)
            return

        # Info entrenamiento
        info_card = Card(scroll, title="📊 Modelos entrenados")
        info_card.pack(fill="x", pady=5)
        models_text = f"Modelos: {', '.join(train_result.get('models_trained', []))}"
        if "revenue_r2" in train_result:
            models_text += f"\nPrecisión ingresos (R²): {train_result['revenue_r2']:.3f}"
        if "demand_r2" in train_result:
            models_text += f"\nPrecisión demanda (R²): {train_result['demand_r2']:.3f}"
        ctk.CTkLabel(info_card, text=models_text, font=AppStyles.FONT_BODY,
                     text_color=COLORS["success"]).pack(padx=16, pady=10)

        # HEALTH SCORE
        hs = self.ai_engine.health_score()
        hs_card = Card(scroll, title="🏥 Health Score Financiero")
        hs_card.pack(fill="x", pady=5)

        score_color = COLORS["success"] if hs["score"] >= 60 else COLORS["warning"] if hs["score"] >= 40 else COLORS["danger"]
        hs_inner = ctk.CTkFrame(hs_card, fg_color="transparent")
        hs_inner.pack(fill="x", padx=16, pady=8)

        ctk.CTkLabel(hs_inner, text=f"{hs['score']:.0f}/100",
                     font=("Segoe UI", 36, "bold"), text_color=score_color).pack(side="left", padx=20)
        ctk.CTkLabel(hs_inner, text=f"Nivel: {hs['nivel']}",
                     font=("Segoe UI", 18, "bold"), text_color=score_color).pack(side="left", padx=10)

        for d in hs.get("details", []):
            bar_frame = ctk.CTkFrame(hs_card, fg_color="transparent")
            bar_frame.pack(fill="x", padx=16, pady=2)
            ctk.CTkLabel(bar_frame, text=f"{d['factor']}: {d['valor']}", font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_light"], width=200).pack(side="left")
            pct = d["score"] / d["max"]
            bar_color = COLORS["success"] if pct > 0.7 else COLORS["warning"] if pct > 0.4 else COLORS["danger"]
            bar = ctk.CTkProgressBar(bar_frame, width=300, height=14,
                                      progress_color=bar_color, fg_color=COLORS["input_bg"])
            bar.set(pct)
            bar.pack(side="left", padx=10)
            ctk.CTkLabel(bar_frame, text=f"{d['score']:.0f}/{d['max']}", font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_muted"]).pack(side="left")
        ctk.CTkLabel(hs_card, text="").pack(pady=4)

        # PREDICCIONES
        pred_card = Card(scroll, title="🔮 Predicción de Ingresos (6 meses)")
        pred_card.pack(fill="x", pady=5)
        preds = self.ai_engine.predict_revenue(6)
        if preds and HAS_MATPLOTLIB:
            fig, ax = plt.subplots(figsize=(9, 3.5), facecolor=COLORS["bg_dark"])
            ax.set_facecolor(COLORS["bg_dark"])
            meses = [p["mes"] for p in preds]
            valores = [p["ingresos_predichos"] for p in preds]
            confianzas = [p["confianza"] for p in preds]

            bars = ax.bar(meses, valores, color=COLORS["accent"], alpha=0.8, edgecolor='white', linewidth=0.5)
            for i, (bar, conf) in enumerate(zip(bars, confianzas)):
                ax.text(bar.get_x() + bar.get_width()/2, bar.get_height() + 100,
                        f'{conf}%', ha='center', color=COLORS["text_muted"], fontsize=8)

            ax.set_ylabel('€', color='white')
            ax.set_title('Ingresos predichos por IA', color='white', fontsize=12, pad=10)
            ax.tick_params(colors='white', labelsize=8)
            for sp in ['top', 'right']: ax.spines[sp].set_visible(False)
            for sp in ['bottom', 'left']: ax.spines[sp].set_color(COLORS["card_border"])
            plt.tight_layout()
            canvas = FigureCanvasTkAgg(fig, master=pred_card)
            canvas.draw()
            canvas.get_tk_widget().pack(fill="x", padx=16, pady=8)
            plt.close(fig)

        # OPTIMIZACIÓN DE PRECIOS
        opt_card = Card(scroll, title="💶 Optimización de Precios (IA)")
        opt_card.pack(fill="x", pady=5)
        optimizations = self.ai_engine.optimize_prices()
        if optimizations:
            hdr = ctk.CTkFrame(opt_card, fg_color=COLORS["bg_light"], corner_radius=6)
            hdr.pack(fill="x", padx=16, pady=(8, 2))
            for txt, w in [("Plato", 180), ("Actual", 70), ("Sugerido", 80), ("Cambio", 70), ("Mejora benef.", 100)]:
                ctk.CTkLabel(hdr, text=txt, font=("Segoe UI", 10, "bold"),
                             text_color=COLORS["gold"], width=w).pack(side="left", padx=2, pady=5)

            for opt in optimizations[:15]:
                row = ctk.CTkFrame(opt_card, fg_color="transparent")
                row.pack(fill="x", padx=16, pady=1)
                cambio_color = COLORS["success"] if opt["cambio_pct"] > 0 else COLORS["danger"]
                vals = [
                    (opt["plato"][:25], 180, COLORS["text_white"]),
                    (f"{opt['precio_actual']:.2f}€", 70, COLORS["text_light"]),
                    (f"{opt['precio_sugerido']:.2f}€", 80, cambio_color),
                    (f"{'+' if opt['cambio_pct']>0 else ''}{opt['cambio_pct']:.1f}%", 70, cambio_color),
                    (f"{'+' if opt['mejora_pct']>0 else ''}{opt['mejora_pct']:.1f}%", 100,
                     COLORS["success"] if opt["mejora_pct"] > 0 else COLORS["text_muted"]),
                ]
                for text, w, color in vals:
                    ctk.CTkLabel(row, text=text, font=AppStyles.FONT_SMALL,
                                 text_color=color, width=w).pack(side="left", padx=2)
            ctk.CTkLabel(opt_card, text="").pack(pady=4)

        # WHAT-IF interactivo
        whatif_card = Card(scroll, title="🔮 Simulación What-If")
        whatif_card.pack(fill="x", pady=5)

        ctk.CTkLabel(whatif_card, text="Ajusta los parámetros y mira el impacto:",
                     font=AppStyles.FONT_BODY, text_color=COLORS["text_light"]).pack(padx=16, pady=(8, 4))

        sliders_frame = ctk.CTkFrame(whatif_card, fg_color="transparent")
        sliders_frame.pack(fill="x", padx=16, pady=4)

        self.whatif_sliders = {}
        for label, key, rango in [("Cambio precios (%)", "precio", (-20, 30)),
                                   ("Cambio clientes (%)", "clientes", (-30, 50)),
                                   ("Cambio coste MP (%)", "mp", (-30, 30))]:
            row = ctk.CTkFrame(sliders_frame, fg_color="transparent")
            row.pack(fill="x", pady=3)
            ctk.CTkLabel(row, text=label, font=AppStyles.FONT_SMALL,
                         text_color=COLORS["text_light"], width=170).pack(side="left")

            val_label = ctk.CTkLabel(row, text="0%", font=AppStyles.FONT_SMALL,
                                      text_color=COLORS["gold"], width=50)
            val_label.pack(side="right")

            slider = ctk.CTkSlider(row, from_=rango[0], to=rango[1], width=300,
                                    fg_color=COLORS["input_bg"], progress_color=COLORS["accent"],
                                    button_color=COLORS["gold"])
            slider.set(0)
            slider.configure(command=lambda v, vl=val_label: vl.configure(text=f"{v:+.0f}%"))
            slider.pack(side="left", padx=10, fill="x", expand=True)
            self.whatif_sliders[key] = slider

        StyledButton(whatif_card, text="🔮 Simular", command=self._run_whatif,
                     style="gold", width=150).pack(padx=16, pady=8)

        self.whatif_result_frame = ctk.CTkFrame(whatif_card, fg_color="transparent")
        self.whatif_result_frame.pack(fill="x", padx=16, pady=(0, 12))

    def _run_whatif(self):
        """Ejecuta simulación what-if con los sliders."""
        if not self.ai_engine:
            return
        for w in self.whatif_result_frame.winfo_children():
            w.destroy()

        changes = {
            "precio_cambio_pct": self.whatif_sliders["precio"].get(),
            "clientes_cambio_pct": self.whatif_sliders["clientes"].get(),
            "costes_mp_cambio_pct": self.whatif_sliders["mp"].get(),
        }
        r = self.ai_engine.whatif_simulation(changes)

        results = [
            ("Ingresos", r["ingresos_antes"], r["ingresos_despues"]),
            ("Costes", r["costes_antes"], r["costes_despues"]),
            ("Beneficio", r["beneficio_antes"], r["beneficio_despues"]),
            ("Margen %", r["margen_antes"], r["margen_despues"]),
        ]

        for label, antes, despues in results:
            row = ctk.CTkFrame(self.whatif_result_frame, fg_color="transparent")
            row.pack(fill="x", pady=2)
            is_pct = "%" in label
            fmt = "{:.1f}%" if is_pct else "{:,.0f}€"
            color = COLORS["success"] if despues > antes else COLORS["danger"]
            ctk.CTkLabel(row, text=f"{label}:", font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_light"], width=120).pack(side="left")
            ctk.CTkLabel(row, text=fmt.format(antes), font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_muted"], width=120).pack(side="left")
            ctk.CTkLabel(row, text="→", font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_muted"]).pack(side="left", padx=10)
            ctk.CTkLabel(row, text=fmt.format(despues), font=("Segoe UI", 13, "bold"),
                         text_color=color, width=120).pack(side="left")

        impacto_color = COLORS["success"] if r["impacto"] > 0 else COLORS["danger"]
        ctk.CTkLabel(self.whatif_result_frame,
                     text=f"\nImpacto mensual: {'+' if r['impacto']>0 else ''}{r['impacto']:,.0f}€",
                     font=("Segoe UI", 15, "bold"), text_color=impacto_color).pack(anchor="w", pady=5)

    # ============================================================
    # FUNCIONES DE DATOS
    # ============================================================

    def _add_pdf(self):
        """Diálogo para añadir PDF."""
        filepaths = filedialog.askopenfilenames(
            title="Seleccionar PDFs",
            filetypes=[("PDF files", "*.pdf"), ("Todos", "*.*")]
        )
        for fp in filepaths:
            self.uploaded_files.append(fp)
            fname = os.path.basename(fp)
            lbl = ctk.CTkLabel(self.pdf_listbox_frame,
                                text=f"📄 {fname}",
                                font=AppStyles.FONT_BODY,
                                text_color=COLORS["text_light"])
            lbl.pack(anchor="w", pady=2)

    def _clear_pdfs(self):
        self.uploaded_files = []
        for w in self.pdf_listbox_frame.winfo_children():
            w.destroy()

    def _extract_pdfs(self):
        """Extrae datos de los PDFs subidos."""
        if not self.uploaded_files:
            messagebox.showwarning("Sin archivos", "No has subido ningún PDF.")
            return

        self.extracted_data = {"platos": [], "facturas": [], "tpv": []}

        for fp in self.uploaded_files:
            result = self.pdf_extractor.extract_from_pdf(fp)
            if "error" in result:
                messagebox.showerror("Error", f"Error en {os.path.basename(fp)}:\n{result['error']}")
                continue

            tipo = result.get("tipo", "generico")
            if tipo == "menu":
                self.extracted_data["platos"].extend(result.get("platos", []))
            elif tipo == "factura":
                self.extracted_data["facturas"].append(result)
            elif tipo == "tpv":
                self.extracted_data["tpv"].append(result)

        # Recoger datos extra
        self._collect_pdf_extra_data()

        messagebox.showinfo("Extracción completada",
                            f"Se han extraído:\n"
                            f"• {len(self.extracted_data['platos'])} platos\n"
                            f"• {len(self.extracted_data['facturas'])} facturas\n"
                            f"• {len(self.extracted_data['tpv'])} resúmenes TPV\n\n"
                            "Revisa los datos antes de continuar.")

        self._show_review()

    def _collect_pdf_extra_data(self):
        """Recoge datos complementarios del formulario PDF."""
        try:
            self.datos.nombre_restaurante = self.pdf_extra_entries.get("nombre", ctk.CTkEntry(self)).get() or "Mi Restaurante"
            self.datos.tipo_restaurante = self.pdf_extra_entries.get("tipo", ctk.CTkEntry(self)).get() or "tradicional"
            self.datos.asientos = int(self.pdf_extra_entries.get("asientos", ctk.CTkEntry(self)).get() or 0)
            self.datos.dias_apertura_mes = int(self.pdf_extra_entries.get("dias_mes", ctk.CTkEntry(self)).get() or 26)
            self.datos.servicios_por_dia = int(self.pdf_extra_entries.get("servicios_dia", ctk.CTkEntry(self)).get() or 2)
            self.datos.clientes_mes = int(self.pdf_extra_entries.get("clientes_mes", ctk.CTkEntry(self)).get() or 0)
            self.datos.alquiler = float(self.pdf_extra_entries.get("alquiler", ctk.CTkEntry(self)).get() or 0)
            self.datos.salarios_fijos = float(self.pdf_extra_entries.get("salarios", ctk.CTkEntry(self)).get() or 0)
            self.datos.seguridad_social = float(self.pdf_extra_entries.get("seg_social", ctk.CTkEntry(self)).get() or 0)
            self.datos.seguros = float(self.pdf_extra_entries.get("seguros", ctk.CTkEntry(self)).get() or 0)
            self.datos.suministros_fijos = float(self.pdf_extra_entries.get("suministros", ctk.CTkEntry(self)).get() or 0)
            self.datos.amortizaciones = float(self.pdf_extra_entries.get("amortizaciones", ctk.CTkEntry(self)).get() or 0)
        except (ValueError, AttributeError):
            pass

    def _get_float(self, key, entries_dict=None):
        """Obtiene valor float de un entry."""
        d = entries_dict or self.manual_entries
        try:
            val = d.get(key)
            if val is None:
                return 0.0
            return float(val.get().replace(',', '.').strip() or 0)
        except (ValueError, AttributeError):
            return 0.0

    def _get_int(self, key, entries_dict=None):
        """Obtiene valor int de un entry."""
        d = entries_dict or self.manual_entries
        try:
            val = d.get(key)
            if val is None:
                return 0
            return int(float(val.get().replace(',', '.').strip() or 0))
        except (ValueError, AttributeError):
            return 0

    def _save_manual_data(self):
        """Guarda los datos del formulario manual."""
        self.datos.nombre_restaurante = self.manual_entries.get("nombre", ctk.CTkEntry(self)).get() or "Mi Restaurante"
        self.datos.tipo_restaurante = self.manual_entries.get("tipo", ctk.CTkEntry(self)).get() or "tradicional"
        self.datos.asientos = self._get_int("asientos")
        self.datos.personal_total = self._get_int("personal")
        self.datos.dias_apertura_mes = self._get_int("dias_mes") or 26
        self.datos.servicios_por_dia = self._get_int("servicios_dia") or 2
        self.datos.clientes_mes = self._get_int("clientes_mes")

        # Costes fijos
        self.datos.alquiler = self._get_float("alquiler")
        self.datos.salarios_fijos = self._get_float("salarios_fijos")
        self.datos.seguridad_social = self._get_float("seg_social")
        self.datos.seguros = self._get_float("seguros")
        self.datos.suministros_fijos = self._get_float("suministros_fijos")
        self.datos.amortizaciones = self._get_float("amortizaciones")
        self.datos.otros_fijos = self._get_float("otros_fijos")

        # Costes variables
        self.datos.coste_mp_total = self._get_float("coste_mp_total")
        self.datos.salarios_variables = self._get_float("salarios_variables")
        self.datos.suministros_variables = self._get_float("suministros_var")
        self.datos.otros_variables = self._get_float("otros_variables")

        # Stock
        self.datos.stock_inicial = self._get_float("stock_inicial")
        self.datos.stock_final = self._get_float("stock_final")
        self.datos.compras_periodo = self._get_float("compras_periodo")

        # Platos
        self.datos.platos = []
        for row in self.platos_rows:
            entries = row["entries"]
            nombre = entries["nombre"].get().strip()
            if not nombre:
                continue
            try:
                precio = float(entries["precio"].get().replace(',', '.') or 0)
                coste = float(entries["coste_mp"].get().replace(',', '.') or 0)
                uds = int(float(entries["unidades"].get() or 0))
                cat = entries["categoria"].get()

                plato = Plato(
                    nombre=nombre,
                    precio_venta=precio,
                    coste_mp=coste,
                    unidades_vendidas=uds,
                    categoria=cat
                )
                self.datos.platos.append(plato)
            except ValueError:
                continue

        if not self.datos.platos:
            messagebox.showwarning("Sin platos", "Añade al menos un plato para continuar.")
            return

        messagebox.showinfo("Datos guardados",
                            f"Se han guardado {len(self.datos.platos)} platos y "
                            "todos los datos del restaurante.\n\nRevisa los datos antes de analizar.")
        self._show_review()

    # ============================================================
    # PANTALLA: REVISIÓN DE DATOS
    # ============================================================

    def _show_review(self):
        """Pantalla de revisión antes del análisis."""
        self._clear_main()
        self._highlight_nav("revision")
        self.current_screen = "revision"

        scroll = self._create_scrollable_frame()

        title = ctk.CTkLabel(scroll, text="🔍 Revisión de Datos",
                              font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"])
        title.pack(anchor="w", pady=(0, 5))

        subtitle = ctk.CTkLabel(scroll,
                                 text="Revisa y ajusta los datos antes de lanzar el análisis.",
                                 font=AppStyles.FONT_SUBTITLE,
                                 text_color=COLORS["text_light"])
        subtitle.pack(anchor="w", pady=(0, 15))

        # Datos generales
        card_gen = Card(scroll, title="🏪 Datos Generales")
        card_gen.pack(fill="x", pady=5)

        gen_frame = ctk.CTkFrame(card_gen, fg_color="transparent")
        gen_frame.pack(fill="x", padx=16, pady=(4, 12))

        info_lines = [
            f"Restaurante: {self.datos.nombre_restaurante}",
            f"Tipo: {self.datos.tipo_restaurante}",
            f"Asientos: {self.datos.asientos}",
            f"Clientes/mes: {self.datos.clientes_mes}",
            f"Días apertura: {self.datos.dias_apertura_mes} | Servicios/día: {self.datos.servicios_por_dia}",
        ]
        for line in info_lines:
            ctk.CTkLabel(gen_frame, text=line, font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_light"]).pack(anchor="w", pady=1)

        # Costes
        card_costes = Card(scroll, title="💰 Resumen de Costes Mensuales")
        card_costes.pack(fill="x", pady=5)

        costes_frame = ctk.CTkFrame(card_costes, fg_color="transparent")
        costes_frame.pack(fill="x", padx=16, pady=(4, 12))
        costes_frame.columnconfigure((0, 1), weight=1)

        costes_left = [
            ("Alquiler:", f"{self.datos.alquiler:,.2f}€"),
            ("Salarios fijos:", f"{self.datos.salarios_fijos:,.2f}€"),
            ("Seg. Social:", f"{self.datos.seguridad_social:,.2f}€"),
            ("Seguros:", f"{self.datos.seguros:,.2f}€"),
            ("Total fijos:", f"{self.datos.costes_fijos_total:,.2f}€"),
        ]
        costes_right = [
            ("Materia prima:", f"{self.datos.coste_mp_total:,.2f}€"),
            ("Salarios variables:", f"{self.datos.salarios_variables:,.2f}€"),
            ("Suministros var.:", f"{self.datos.suministros_variables:,.2f}€"),
            ("Total variables:", f"{self.datos.costes_variables_total:,.2f}€"),
            ("TOTAL COSTES:", f"{self.datos.costes_totales:,.2f}€"),
        ]

        for i, (label, value) in enumerate(costes_left):
            color = COLORS["gold"] if "Total" in label else COLORS["text_light"]
            ctk.CTkLabel(costes_frame, text=f"{label} {value}",
                         font=AppStyles.FONT_BODY, text_color=color
                         ).grid(row=i, column=0, sticky="w", padx=5, pady=2)

        for i, (label, value) in enumerate(costes_right):
            color = COLORS["accent"] if "TOTAL" in label else (COLORS["gold"] if "Total" in label else COLORS["text_light"])
            ctk.CTkLabel(costes_frame, text=f"{label} {value}",
                         font=AppStyles.FONT_BODY, text_color=color
                         ).grid(row=i, column=1, sticky="w", padx=5, pady=2)

        # Platos
        card_platos = Card(scroll, title=f"🍽️ Carta ({len(self.datos.platos)} platos)")
        card_platos.pack(fill="x", pady=5)

        if self.datos.platos:
            # Header
            hdr = ctk.CTkFrame(card_platos, fg_color=COLORS["bg_light"], corner_radius=6)
            hdr.pack(fill="x", padx=16, pady=(8, 2))
            for txt, w in [("Plato", 200), ("Cat.", 90), ("PV(€)", 80),
                           ("Coste MP(€)", 90), ("Margen(€)", 90),
                           ("%MP", 60), ("Uds/mes", 70)]:
                ctk.CTkLabel(hdr, text=txt, font=("Segoe UI", 11, "bold"),
                             text_color=COLORS["gold"], width=w).pack(side="left", padx=3, pady=5)

            for plato in self.datos.platos:
                row = ctk.CTkFrame(card_platos, fg_color="transparent")
                row.pack(fill="x", padx=16, pady=1)

                margen_color = COLORS["success"] if plato.pct_coste_mp <= 40 else COLORS["warning"] if plato.pct_coste_mp <= 50 else COLORS["danger"]

                vals = [
                    (plato.nombre, 200, COLORS["text_white"]),
                    (plato.categoria, 90, COLORS["text_muted"]),
                    (f"{plato.precio_venta:.2f}", 80, COLORS["text_light"]),
                    (f"{plato.coste_mp:.2f}", 90, COLORS["text_light"]),
                    (f"{plato.margen_bruto:.2f}", 90, COLORS["success"]),
                    (f"{plato.pct_coste_mp:.1f}%", 60, margen_color),
                    (f"{plato.unidades_vendidas}", 70, COLORS["text_light"]),
                ]
                for text, w, color in vals:
                    ctk.CTkLabel(row, text=text, font=AppStyles.FONT_BODY,
                                 text_color=color, width=w).pack(side="left", padx=3)
        else:
            ctk.CTkLabel(card_platos, text="No hay platos cargados. Vuelve a la sección de datos.",
                         font=AppStyles.FONT_BODY, text_color=COLORS["warning"]
                         ).pack(padx=16, pady=12)

        # Botones de acción
        action_frame = ctk.CTkFrame(scroll, fg_color="transparent")
        action_frame.pack(fill="x", pady=15)

        StyledButton(action_frame, text="← Editar datos",
                     command=self._show_data_input,
                     style="secondary", width=160).pack(side="left", padx=5)

        StyledButton(action_frame, text="🚀 Lanzar Análisis →",
                     command=self._run_analysis,
                     style="gold", width=200).pack(side="right", padx=5)

    # ============================================================
    # EJECUCIÓN DEL ANÁLISIS
    # ============================================================

    def _run_analysis(self):
        """Ejecuta el análisis financiero completo."""
        if not self.datos.platos:
            messagebox.showwarning("Sin datos", "Introduce datos antes de analizar.")
            return

        self.analisis = AnalisisFinanciero(self.datos)

        # Entrenar IA si es Pro
        if self.is_pro and self.ai_engine and HAS_AI:
            self.ai_engine.datos = self.datos
            self.ai_engine.train_models()

        self.chatbot = RestaurantChatbot(self.datos, self.analisis,
                                          self.ai_engine if self.is_pro else None)
        self._show_analysis()

    # ============================================================
    # PANTALLA: RESULTADOS DEL ANÁLISIS
    # ============================================================

    def _show_analysis(self):
        """Pantalla de resultados del análisis."""
        if not self.analisis:
            messagebox.showinfo("Sin análisis", "Primero introduce datos y lanza el análisis.")
            self._show_data_input()
            return

        self._clear_main()
        self._highlight_nav("analisis")
        self.current_screen = "analisis"

        scroll = self._create_scrollable_frame()

        title = ctk.CTkLabel(scroll, text=f"📊 Análisis Financiero — {self.datos.nombre_restaurante}",
                              font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"])
        title.pack(anchor="w", pady=(0, 15))

        # KPIs principales
        costes = self.analisis.analisis_costes()
        cp = self.analisis.analisis_corto_plazo()

        kpi_frame = ctk.CTkFrame(scroll, fg_color="transparent")
        kpi_frame.pack(fill="x", pady=5)
        kpi_frame.columnconfigure((0, 1, 2, 3, 4), weight=1)

        kpis = [
            ("Ingresos/mes", f"{costes['ingresos']:,.0f}", "€", COLORS["text_white"]),
            ("Beneficio neto", f"{costes['beneficio_neto']:,.0f}", "€",
             COLORS["success"] if costes["beneficio_neto"] > 0 else COLORS["danger"]),
            ("Margen neto", f"{costes['margen_neto_pct']:.1f}", "%",
             COLORS["success"] if costes["margen_neto_pct"] > 10 else COLORS["warning"]),
            ("Ticket medio", f"{cp['ticket_medio']:.2f}", "€", COLORS["gold"]),
            ("Cash flow", f"{cp['cash_flow_mensual']:,.0f}", "€",
             COLORS["success"] if cp["cash_flow_mensual"] > 0 else COLORS["danger"]),
        ]

        for i, (label, value, unit, color) in enumerate(kpis):
            mc = MetricCard(kpi_frame, label=label, value=value, unit=unit, color=color)
            mc.grid(row=0, column=i, padx=4, pady=4, sticky="nsew")

        # Alertas
        alertas = costes.get("alertas", []) + cp.get("alertas", [])
        if alertas:
            alert_card = Card(scroll, title="⚠️ Alertas")
            alert_card.pack(fill="x", pady=5)
            for alerta in alertas:
                ctk.CTkLabel(alert_card, text=alerta, font=AppStyles.FONT_BODY,
                             text_color=COLORS["warning"], wraplength=800,
                             justify="left").pack(padx=16, pady=3, anchor="w")
            ctk.CTkLabel(alert_card, text="").pack(pady=2)

        # ESTRUCTURA DE COSTES
        costes_card = Card(scroll, title="💰 Estructura de Costes")
        costes_card.pack(fill="x", pady=5)

        costes_inner = ctk.CTkFrame(costes_card, fg_color="transparent")
        costes_inner.pack(fill="x", padx=16, pady=(4, 12))
        costes_inner.columnconfigure((0, 1), weight=1)

        # Gráfico de costes si matplotlib disponible
        if HAS_MATPLOTLIB:
            fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(9, 3.5), facecolor=COLORS["bg_dark"])

            # Pie chart de costes
            labels_pie = ['Materia Prima', 'Personal', 'Otros fijos', 'Otros variables']
            values_pie = [
                self.datos.coste_mp_total,
                self.datos.salarios_fijos + self.datos.seguridad_social + self.datos.salarios_variables,
                self.datos.alquiler + self.datos.seguros + self.datos.suministros_fijos + self.datos.amortizaciones,
                self.datos.suministros_variables + self.datos.otros_variables + self.datos.otros_fijos,
            ]
            values_pie = [max(0, v) for v in values_pie]
            colors_pie = ['#e94560', '#d4a574', '#3498db', '#2ecc71']

            if sum(values_pie) > 0:
                ax1.pie(values_pie, labels=labels_pie, autopct='%1.1f%%',
                        colors=colors_pie, textprops={'color': 'white', 'fontsize': 9},
                        pctdistance=0.75)
                ax1.set_title('Distribución de Costes', color='white', fontsize=11, pad=10)
            ax1.set_facecolor(COLORS["bg_dark"])

            # Bar chart de referencia vs real
            categories = ['Mat. Prima', 'Personal', 'Gastos Gen.']
            real_vals = [costes['pct_materia_prima'], costes['pct_personal'],
                         100 - costes['pct_materia_prima'] - costes['pct_personal']]
            ref_vals = [REF_MATERIA_PRIMA * 100, REF_PERSONAL * 100, REF_GASTOS_GENERALES * 100]

            x = range(len(categories))
            ax2.bar([i - 0.15 for i in x], real_vals, 0.3, label='Real', color='#e94560')
            ax2.bar([i + 0.15 for i in x], ref_vals, 0.3, label='Referencia', color='#3498db', alpha=0.7)
            ax2.set_xticks(x)
            ax2.set_xticklabels(categories, color='white', fontsize=9)
            ax2.set_ylabel('%', color='white')
            ax2.tick_params(colors='white')
            ax2.legend(fontsize=8, facecolor=COLORS["card_bg"], edgecolor=COLORS["card_border"],
                       labelcolor='white')
            ax2.set_title('Real vs Referencia', color='white', fontsize=11, pad=10)
            ax2.set_facecolor(COLORS["bg_dark"])
            ax2.spines['bottom'].set_color(COLORS["card_border"])
            ax2.spines['left'].set_color(COLORS["card_border"])
            ax2.spines['top'].set_visible(False)
            ax2.spines['right'].set_visible(False)

            plt.tight_layout()
            canvas = FigureCanvasTkAgg(fig, master=costes_inner)
            canvas.draw()
            canvas.get_tk_widget().pack(fill="x", pady=5)
            plt.close(fig)

        # PUNTO MUERTO
        pm = self.analisis.punto_muerto()
        pm_card = Card(scroll, title="📈 Punto Muerto / Umbral de Rentabilidad")
        pm_card.pack(fill="x", pady=5)

        if "error" not in pm:
            pm_inner = ctk.CTkFrame(pm_card, fg_color="transparent")
            pm_inner.pack(fill="x", padx=16, pady=(4, 12))
            pm_inner.columnconfigure((0, 1, 2, 3), weight=1)

            pm_data = [
                ("PM (platos)", f"{pm['punto_muerto_unidades']:.0f}", COLORS["accent"]),
                ("PM (euros)", f"{pm['punto_muerto_euros']:,.0f}€", COLORS["accent"]),
                ("Margen seguridad", f"{pm['margen_seguridad_pct']:.1f}%",
                 COLORS["success"] if pm["margen_seguridad_pct"] > 20 else COLORS["warning"]),
                ("Estado", "✅ Superado" if pm["superado"] else "❌ No alcanzado",
                 COLORS["success"] if pm["superado"] else COLORS["danger"]),
            ]
            for i, (label, value, color) in enumerate(pm_data):
                mc = MetricCard(pm_inner, label=label, value=value, color=color)
                mc.grid(row=0, column=i, padx=4, pady=4, sticky="nsew")
        else:
            ctk.CTkLabel(pm_card, text=pm["error"], font=AppStyles.FONT_BODY,
                         text_color=COLORS["warning"]).pack(padx=16, pady=10)

        # MENU ENGINEERING
        me = self.analisis.menu_engineering()
        me_card = Card(scroll, title="🍽️ Menu Engineering")
        me_card.pack(fill="x", pady=5)

        if "clasificacion" in me:
            # Resumen por categoría
            cat_frame = ctk.CTkFrame(me_card, fg_color="transparent")
            cat_frame.pack(fill="x", padx=16, pady=8)
            cat_frame.columnconfigure((0, 1, 2, 3), weight=1)

            for i, (cat_key, cat_data) in enumerate(me["resumen"].items()):
                cat_info = MENU_CATEGORIES[cat_key]
                f = ctk.CTkFrame(cat_frame, fg_color=COLORS["card_bg"],
                                  corner_radius=8, border_width=2,
                                  border_color=cat_info["color"])
                f.grid(row=0, column=i, padx=4, pady=4, sticky="nsew")

                ctk.CTkLabel(f, text=cat_info["label"],
                             font=("Segoe UI", 14, "bold"),
                             text_color=cat_info["color"]).pack(pady=(8, 2))
                ctk.CTkLabel(f, text=f"{cat_data['count']} platos",
                             font=AppStyles.FONT_BODY,
                             text_color=COLORS["text_light"]).pack()
                ctk.CTkLabel(f, text=f"Beneficio: {cat_data['beneficio']:,.0f}€",
                             font=AppStyles.FONT_SMALL,
                             text_color=COLORS["text_muted"]).pack(pady=(0, 8))

            # Tabla detallada
            hdr = ctk.CTkFrame(me_card, fg_color=COLORS["bg_light"], corner_radius=6)
            hdr.pack(fill="x", padx=16, pady=(10, 2))
            for txt, w in [("Plato", 180), ("Categoría", 110), ("PV(€)", 70),
                           ("Margen(€)", 80), ("Uds", 50), ("IP(%)", 60),
                           ("IM", 50), ("Acción", 200)]:
                ctk.CTkLabel(hdr, text=txt, font=("Segoe UI", 10, "bold"),
                             text_color=COLORS["gold"], width=w).pack(side="left", padx=2, pady=5)

            for item in me["clasificacion"]:
                row = ctk.CTkFrame(me_card, fg_color="transparent")
                row.pack(fill="x", padx=16, pady=1)

                cat_info = item["categoria_info"]
                vals = [
                    (item["plato"], 180, COLORS["text_white"]),
                    (cat_info["label"], 110, cat_info["color"]),
                    (f"{item['precio']:.2f}", 70, COLORS["text_light"]),
                    (f"{item['margen_bruto']:.2f}", 80, COLORS["text_light"]),
                    (f"{item['unidades']}", 50, COLORS["text_light"]),
                    (f"{item['indice_popularidad']:.1f}", 60, COLORS["text_light"]),
                    (f"{item['indice_margen']:.2f}", 50, COLORS["text_light"]),
                    (cat_info["action"], 200, cat_info["color"]),
                ]
                for text, w, color in vals:
                    ctk.CTkLabel(row, text=text, font=AppStyles.FONT_SMALL,
                                 text_color=color, width=w).pack(side="left", padx=2)

            # Gráfico Menu Engineering
            if HAS_MATPLOTLIB and me["clasificacion"]:
                fig_me, ax = plt.subplots(figsize=(8, 4), facecolor=COLORS["bg_dark"])
                ax.set_facecolor(COLORS["bg_dark"])

                for item in me["clasificacion"]:
                    color = item["categoria_info"]["color"]
                    ax.scatter(item["indice_popularidad"], item["margen_bruto"],
                               c=color, s=item["unidades"] * 3 + 50, alpha=0.8,
                               edgecolors='white', linewidths=0.5)
                    ax.annotate(item["plato"][:15], (item["indice_popularidad"], item["margen_bruto"]),
                                fontsize=7, color='white', ha='center', va='bottom')

                ax.axhline(y=me["margen_medio"], color=COLORS["gold"], linestyle='--',
                           alpha=0.6, label=f'Margen medio: {me["margen_medio"]:.2f}€')
                ax.axvline(x=me["ip_umbral"], color=COLORS["accent"], linestyle='--',
                           alpha=0.6, label=f'IP umbral: {me["ip_umbral"]:.1f}%')

                ax.set_xlabel('Índice de Popularidad (%)', color='white', fontsize=10)
                ax.set_ylabel('Margen Bruto (€)', color='white', fontsize=10)
                ax.set_title('Mapa de Menu Engineering', color='white', fontsize=12, pad=10)
                ax.tick_params(colors='white')
                ax.legend(fontsize=8, facecolor=COLORS["card_bg"],
                          edgecolor=COLORS["card_border"], labelcolor='white')
                ax.spines['bottom'].set_color(COLORS["card_border"])
                ax.spines['left'].set_color(COLORS["card_border"])
                ax.spines['top'].set_visible(False)
                ax.spines['right'].set_visible(False)

                # Cuadrantes
                ax.text(0.75, 0.85, 'ESTRELLA ⭐', transform=ax.transAxes, color='#2ecc71',
                        fontsize=9, alpha=0.4, ha='center', fontweight='bold')
                ax.text(0.75, 0.15, 'VACA 🐄', transform=ax.transAxes, color='#f39c12',
                        fontsize=9, alpha=0.4, ha='center', fontweight='bold')
                ax.text(0.25, 0.85, 'PUZZLE 🧩', transform=ax.transAxes, color='#3498db',
                        fontsize=9, alpha=0.4, ha='center', fontweight='bold')
                ax.text(0.25, 0.15, 'PERRO 🐕', transform=ax.transAxes, color='#e74c3c',
                        fontsize=9, alpha=0.4, ha='center', fontweight='bold')

                plt.tight_layout()
                canvas_me = FigureCanvasTkAgg(fig_me, master=me_card)
                canvas_me.draw()
                canvas_me.get_tk_widget().pack(fill="x", padx=16, pady=8)
                plt.close(fig_me)

            # Recomendaciones
            if me.get("recomendaciones"):
                rec_card = Card(scroll, title="💡 Recomendaciones Menu Engineering")
                rec_card.pack(fill="x", pady=5)
                for rec in me["recomendaciones"]:
                    ctk.CTkLabel(rec_card, text=rec, font=AppStyles.FONT_BODY,
                                 text_color=COLORS["text_light"], wraplength=800,
                                 justify="left").pack(padx=16, pady=4, anchor="w")
                ctk.CTkLabel(rec_card, text="").pack(pady=2)

        # Export buttons (PRO)
        if self.is_pro:
            export_frame = ctk.CTkFrame(scroll, fg_color="transparent")
            export_frame.pack(fill="x", pady=10)
            ctk.CTkLabel(export_frame, text="📤 Exportar informe:",
                         font=AppStyles.FONT_HEADING,
                         text_color=COLORS["gold"]).pack(side="left", padx=(0, 15))
            StyledButton(export_frame, text="📄 PDF", command=self._export_pdf,
                         style="primary", width=120).pack(side="left", padx=5)
            StyledButton(export_frame, text="📊 Excel", command=self._export_excel,
                         style="success", width=120).pack(side="left", padx=5)

        # ANÁLISIS PRO
        if self.is_pro:
            self._show_pro_analysis(scroll)

            # Botones de exportación (PRO)
            export_frame = ctk.CTkFrame(scroll, fg_color="transparent")
            export_frame.pack(fill="x", pady=15)

            ctk.CTkLabel(export_frame, text="📤 Exportar informe:",
                         font=AppStyles.FONT_HEADING,
                         text_color=COLORS["gold"]).pack(side="left", padx=10)

            StyledButton(export_frame, text="📄 Exportar PDF",
                         command=self._export_pdf,
                         style="primary", width=160).pack(side="left", padx=5)
            StyledButton(export_frame, text="📊 Exportar Excel",
                         command=self._export_excel,
                         style="success", width=160).pack(side="left", padx=5)
        else:
            # Banner PRO
            pro_banner = ctk.CTkFrame(scroll, fg_color=COLORS["bg_light"],
                                       corner_radius=12, border_width=2,
                                       border_color=COLORS["gold"])
            pro_banner.pack(fill="x", pady=15)
            ctk.CTkLabel(pro_banner, text="🔒 Análisis avanzados disponibles en la versión PRO",
                         font=("Segoe UI", 16, "bold"),
                         text_color=COLORS["gold"]).pack(pady=(15, 5))
            ctk.CTkLabel(pro_banner, text="Incluye: Análisis a medio y largo plazo, simulaciones, "
                         "proyecciones a 3 años,\nchatbot personalizado, exportación PDF/Excel y soporte continuo.",
                         font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_light"]).pack(pady=(0, 5))
            StyledButton(pro_banner, text="🔑 Activar versión PRO →",
                         command=self._show_license,
                         style="gold", width=220).pack(pady=(5, 15))

    def _show_pro_analysis(self, scroll):
        """Muestra análisis avanzados (solo PRO)."""
        mp = self.analisis.analisis_medio_plazo()
        lp = self.analisis.analisis_largo_plazo()

        # MEDIO PLAZO
        mp_card = Card(scroll, title="📅 Análisis a Medio Plazo (PRO)")
        mp_card.pack(fill="x", pady=5)

        mp_inner = ctk.CTkFrame(mp_card, fg_color="transparent")
        mp_inner.pack(fill="x", padx=16, pady=(4, 12))

        ctk.CTkLabel(mp_inner, text=f"Proyección trimestral: {mp['ingresos_proyectados_trim']:,.0f}€ ingresos | "
                     f"{mp['beneficio_proyectado_trim']:,.0f}€ beneficio",
                     font=AppStyles.FONT_BODY, text_color=COLORS["text_light"]).pack(anchor="w", pady=4)

        # Simulaciones
        ctk.CTkLabel(mp_inner, text="Simulaciones de mejora:",
                     font=("Segoe UI", 13, "bold"), text_color=COLORS["gold"]).pack(anchor="w", pady=(8, 4))

        for sim in mp.get("simulaciones", []):
            sim_text = f"📌 {sim['nombre']} → Nuevo margen: {sim['nuevo_margen']:.1f}%"
            color = COLORS["success"] if sim["nuevo_margen"] > 15 else COLORS["warning"]
            ctk.CTkLabel(mp_inner, text=sim_text, font=AppStyles.FONT_BODY,
                         text_color=color).pack(anchor="w", pady=2)

        # LARGO PLAZO
        lp_card = Card(scroll, title="🏗️ Análisis a Largo Plazo (PRO)")
        lp_card.pack(fill="x", pady=5)

        lp_inner = ctk.CTkFrame(lp_card, fg_color="transparent")
        lp_inner.pack(fill="x", padx=16, pady=(4, 12))
        lp_inner.columnconfigure((0, 1, 2, 3), weight=1)

        lp_kpis = [
            ("Beneficio anual", f"{lp['beneficio_anual']:,.0f}€",
             COLORS["success"] if lp["beneficio_anual"] > 0 else COLORS["danger"]),
            ("Rentabilidad", f"{lp['rentabilidad_economica']:.1f}%", COLORS["gold"]),
            ("ROI", f"{lp['roi']:.1f}%", COLORS["text_white"]),
            ("Payback", f"{lp['payback_meses']:.0f} meses" if lp["payback_meses"] < 999 else "N/A",
             COLORS["warning"] if lp["payback_meses"] > 36 else COLORS["success"]),
        ]

        for i, (label, value, color) in enumerate(lp_kpis):
            mc = MetricCard(lp_inner, label=label, value=value, color=color)
            mc.grid(row=0, column=i, padx=4, pady=4, sticky="nsew")

        # Salud financiera
        salud = lp["salud_financiera"]
        salud_color = {"danger": COLORS["danger"], "warning": COLORS["warning"],
                       "gold": COLORS["gold"], "success": COLORS["success"]}
        ctk.CTkLabel(lp_inner, text=f"\n🏥 Salud financiera: {salud['estado'].upper()}",
                     font=("Segoe UI", 14, "bold"),
                     text_color=salud_color.get(salud["color"], COLORS["text_light"])
                     ).grid(row=1, column=0, columnspan=4, sticky="w", padx=5, pady=8)
        ctk.CTkLabel(lp_inner, text=salud["mensaje"],
                     font=AppStyles.FONT_BODY,
                     text_color=COLORS["text_light"]
                     ).grid(row=2, column=0, columnspan=4, sticky="w", padx=5)

        # Proyección a 3 años
        if HAS_MATPLOTLIB and lp.get("proyeccion_3_años"):
            fig_lp, ax = plt.subplots(figsize=(8, 3.5), facecolor=COLORS["bg_dark"])
            ax.set_facecolor(COLORS["bg_dark"])

            años = [f"Año {p['año']}" for p in lp["proyeccion_3_años"]]
            ingresos = [p["ingresos"] for p in lp["proyeccion_3_años"]]
            costes_lp = [p["costes"] for p in lp["proyeccion_3_años"]]
            beneficios = [p["beneficio"] for p in lp["proyeccion_3_años"]]

            x = range(len(años))
            ax.bar([i - 0.25 for i in x], [v / 1000 for v in ingresos], 0.25,
                   label='Ingresos', color='#3498db')
            ax.bar(x, [v / 1000 for v in costes_lp], 0.25,
                   label='Costes', color='#e94560')
            ax.bar([i + 0.25 for i in x], [v / 1000 for v in beneficios], 0.25,
                   label='Beneficio', color='#2ecc71')

            ax.set_xticks(x)
            ax.set_xticklabels(años, color='white')
            ax.set_ylabel('Miles €', color='white')
            ax.set_title('Proyección a 3 años', color='white', fontsize=12, pad=10)
            ax.tick_params(colors='white')
            ax.legend(fontsize=8, facecolor=COLORS["card_bg"],
                      edgecolor=COLORS["card_border"], labelcolor='white')
            ax.spines['bottom'].set_color(COLORS["card_border"])
            ax.spines['left'].set_color(COLORS["card_border"])
            ax.spines['top'].set_visible(False)
            ax.spines['right'].set_visible(False)

            plt.tight_layout()
            canvas_lp = FigureCanvasTkAgg(fig_lp, master=lp_card)
            canvas_lp.draw()
            canvas_lp.get_tk_widget().pack(fill="x", padx=16, pady=8)
            plt.close(fig_lp)

        # Plan de acción
        plan_card = Card(scroll, title="📋 Plan de Acción Estratégico")
        plan_card.pack(fill="x", pady=5)
        for accion in lp.get("plan_accion", []):
            ctk.CTkLabel(plan_card, text=accion, font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_light"], wraplength=800,
                         justify="left").pack(padx=16, pady=3, anchor="w")
        ctk.CTkLabel(plan_card, text="").pack(pady=4)

    # ============================================================
    # EXPORTAR INFORMES
    # ============================================================

    def _export_pdf(self):
        """Exporta informe a PDF."""
        filepath = filedialog.asksaveasfilename(
            title="Guardar informe PDF",
            defaultextension=".pdf",
            filetypes=[("PDF", "*.pdf")],
            initialfile=f"Informe_{self.datos.nombre_restaurante}_{datetime.now().strftime('%Y%m%d')}.pdf"
        )
        if not filepath:
            return

        logo_path = os.path.join(BASE_DIR, "assets", "logo.png")
        exporter = ReportExporter(self.datos, self.analisis,
                                   logo_path if os.path.exists(logo_path) else None)
        result = exporter.export_pdf(filepath)
        if result["success"]:
            messagebox.showinfo("PDF Exportado", f"Informe guardado en:\n{filepath}")
        else:
            messagebox.showerror("Error", f"No se pudo exportar: {result['error']}")

    def _export_excel(self):
        """Exporta informe a Excel."""
        filepath = filedialog.asksaveasfilename(
            title="Guardar informe Excel",
            defaultextension=".xlsx",
            filetypes=[("Excel", "*.xlsx")],
            initialfile=f"Informe_{self.datos.nombre_restaurante}_{datetime.now().strftime('%Y%m%d')}.xlsx"
        )
        if not filepath:
            return

        exporter = ReportExporter(self.datos, self.analisis)
        result = exporter.export_excel(filepath)
        if result["success"]:
            messagebox.showinfo("Excel Exportado", f"Informe guardado en:\n{filepath}")
        else:
            messagebox.showerror("Error", f"No se pudo exportar: {result['error']}")

    # ============================================================
    # EXPORTACIÓN (PRO)
    # ============================================================

    def _export_pdf(self):
        """Exporta el informe a PDF."""
        if not self.analisis:
            messagebox.showwarning("Sin datos", "Ejecuta el análisis primero.")
            return

        filepath = filedialog.asksaveasfilename(
            title="Guardar informe PDF",
            defaultextension=".pdf",
            filetypes=[("PDF", "*.pdf")],
            initialfile=f"Informe_{self.datos.nombre_restaurante}_{datetime.now().strftime('%Y%m%d')}.pdf"
        )
        if not filepath:
            return

        exporter = ReportExporter(self.datos, self.analisis)
        logo_path = os.path.join(BASE_DIR, "assets", "logo.png")
        success = exporter.export_pdf(filepath, logo_path if os.path.exists(logo_path) else None)

        if success:
            messagebox.showinfo("Exportado", f"Informe PDF guardado en:\n{filepath}")
        else:
            messagebox.showerror("Error", "No se pudo exportar el PDF.\nVerifica que reportlab está instalado.")

    def _export_excel(self):
        """Exporta el informe a Excel."""
        if not self.analisis:
            messagebox.showwarning("Sin datos", "Ejecuta el análisis primero.")
            return

        filepath = filedialog.asksaveasfilename(
            title="Guardar informe Excel",
            defaultextension=".xlsx",
            filetypes=[("Excel", "*.xlsx")],
            initialfile=f"Informe_{self.datos.nombre_restaurante}_{datetime.now().strftime('%Y%m%d')}.xlsx"
        )
        if not filepath:
            return

        exporter = ReportExporter(self.datos, self.analisis)
        success = exporter.export_excel(filepath)

        if success:
            messagebox.showinfo("Exportado", f"Informe Excel guardado en:\n{filepath}")
        else:
            messagebox.showerror("Error", "No se pudo exportar el Excel.\nVerifica que openpyxl está instalado.")

    # ============================================================
    # PANTALLA: CHATBOT
    # ============================================================

    def _show_chatbot(self):
        """Pantalla del chatbot asistente."""
        self._clear_main()
        self._highlight_nav("chatbot")
        self.current_screen = "chatbot"

        if not self.is_pro:
            # Banner PRO
            frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
            frame.pack(fill="both", expand=True, padx=40, pady=40)

            ctk.CTkLabel(frame, text="💬 Asistente IA Personalizado",
                         font=AppStyles.FONT_TITLE, text_color=COLORS["text_white"]).pack(pady=(40, 10))
            ctk.CTkLabel(frame, text="🔒 Función exclusiva de la versión PRO",
                         font=AppStyles.FONT_SUBTITLE,
                         text_color=COLORS["gold"]).pack(pady=10)
            ctk.CTkLabel(frame, text="El asistente analiza los datos de TU restaurante y ofrece:\n\n"
                         "• Respuestas personalizadas sobre tus costes, márgenes y platos\n"
                         "• Soporte diario con consejos accionables\n"
                         "• Informe semanal automatizado\n"
                         "• Informe mensual completo",
                         font=AppStyles.FONT_BODY, text_color=COLORS["text_light"],
                         justify="left").pack(pady=15)
            StyledButton(frame, text="🔑 Activar PRO →", command=self._show_license,
                         style="gold", width=200).pack(pady=15)
            return

        if not self.chatbot:
            if self.analisis:
                self.chatbot = RestaurantChatbot(self.datos, self.analisis,
                                                  self.ai_engine if self.is_pro else None)
            else:
                self.chatbot = RestaurantChatbot()

        # Header
        header = ctk.CTkFrame(self.main_area, fg_color="transparent", height=50)
        header.pack(fill="x", padx=20, pady=(20, 0))
        header.pack_propagate(False)

        ctk.CTkLabel(header, text="💬 Asistente RestaurantAI",
                     font=AppStyles.FONT_HEADING,
                     text_color=COLORS["text_white"]).pack(side="left")

        # Botones de soporte
        btn_frame = ctk.CTkFrame(header, fg_color="transparent")
        btn_frame.pack(side="right")

        StyledButton(btn_frame, text="📅 Diario",
                     command=lambda: self._chat_insert(self.chatbot.get_soporte_diario()),
                     style="secondary", width=90).pack(side="left", padx=3)
        StyledButton(btn_frame, text="📊 Semanal",
                     command=lambda: self._chat_insert(self.chatbot.get_soporte_semanal()),
                     style="secondary", width=90).pack(side="left", padx=3)
        StyledButton(btn_frame, text="📈 Mensual",
                     command=lambda: self._chat_insert(self.chatbot.get_soporte_mensual()),
                     style="secondary", width=90).pack(side="left", padx=3)

        # Área de chat
        self.chat_display = ctk.CTkTextbox(
            self.main_area, fg_color=COLORS["card_bg"],
            text_color=COLORS["text_light"],
            font=AppStyles.FONT_MONO,
            corner_radius=10, border_width=1,
            border_color=COLORS["card_border"],
            wrap="word"
        )
        self.chat_display.pack(fill="both", expand=True, padx=20, pady=10)

        # Mensaje de bienvenida
        welcome_msg = self.chatbot.chat("hola")
        self.chat_display.insert("end", f"🤖 Asistente:\n{welcome_msg}\n\n")

        # Input
        input_frame = ctk.CTkFrame(self.main_area, fg_color="transparent", height=50)
        input_frame.pack(fill="x", padx=20, pady=(0, 20))
        input_frame.pack_propagate(False)

        self.chat_input = StyledEntry(input_frame,
                                       placeholder="Escribe tu pregunta aquí...",
                                       width=800)
        self.chat_input.pack(side="left", fill="x", expand=True, padx=(0, 10))
        self.chat_input.bind("<Return>", lambda e: self._send_chat())

        StyledButton(input_frame, text="Enviar →",
                     command=self._send_chat,
                     style="primary", width=100).pack(side="right")

    def _send_chat(self):
        """Envía mensaje al chatbot."""
        msg = self.chat_input.get().strip()
        if not msg:
            return

        self.chat_display.insert("end", f"👤 Tú:\n{msg}\n\n")
        self.chat_input.delete(0, "end")

        response = self.chatbot.chat(msg)
        self.chat_display.insert("end", f"🤖 Asistente:\n{response}\n\n")
        self.chat_display.see("end")

    def _chat_insert(self, text):
        """Inserta texto en el chat."""
        self.chat_display.insert("end", f"🤖 Asistente:\n{text}\n\n")
        self.chat_display.see("end")

    # ============================================================
    # PANTALLA: LICENCIA
    # ============================================================

    def _show_license(self):
        """Pantalla de gestión de licencia."""
        self._clear_main()
        self._highlight_nav("licencia")
        self.current_screen = "licencia"

        frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=40, pady=30)

        ctk.CTkLabel(frame, text="🔑 Gestión de Licencia",
                     font=AppStyles.FONT_TITLE,
                     text_color=COLORS["text_white"]).pack(pady=(0, 20))

        # Estado actual
        status = self.license_mgr.check_license()
        status_card = Card(frame, title="Estado actual")
        status_card.pack(fill="x", pady=10)

        if status.get("valid"):
            ctk.CTkLabel(status_card, text="✅ Licencia PRO activa",
                         font=("Segoe UI", 16, "bold"),
                         text_color=COLORS["success"]).pack(padx=16, pady=5)
            ctk.CTkLabel(status_card, text=f"Expira: {status.get('expires', 'N/A')}\n"
                         f"Días restantes: {status.get('days_left', 'N/A')}",
                         font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_light"]).pack(padx=16, pady=(0, 12))
        else:
            ctk.CTkLabel(status_card, text="🔒 Versión DEMO",
                         font=("Segoe UI", 16, "bold"),
                         text_color=COLORS["warning"]).pack(padx=16, pady=5)
            ctk.CTkLabel(status_card, text="Funciones limitadas. Activa PRO para acceso completo.",
                         font=AppStyles.FONT_BODY,
                         text_color=COLORS["text_light"]).pack(padx=16, pady=(0, 12))

        # Hardware ID
        hw_card = Card(frame, title="ID de este equipo")
        hw_card.pack(fill="x", pady=10)

        hw_id = self.license_mgr.get_hardware_id()
        hw_frame = ctk.CTkFrame(hw_card, fg_color="transparent")
        hw_frame.pack(fill="x", padx=16, pady=8)

        hw_entry = StyledEntry(hw_frame, width=400)
        hw_entry.insert(0, hw_id)
        hw_entry.configure(state="disabled")
        hw_entry.pack(side="left", padx=(0, 10))

        def copy_hw():
            self.clipboard_clear()
            self.clipboard_append(hw_id)
            messagebox.showinfo("Copiado", "ID copiado al portapapeles.\n"
                                "Envíalo a convderover@gmail.com para obtener tu licencia PRO.")

        StyledButton(hw_frame, text="📋 Copiar", command=copy_hw,
                     style="secondary", width=100).pack(side="left")

        ctk.CTkLabel(hw_card, text=f"Envía este ID a {CONTACT_EMAIL} para obtener tu clave PRO.",
                     font=AppStyles.FONT_SMALL,
                     text_color=COLORS["text_muted"]).pack(padx=16, pady=(0, 12))

        # Activar licencia
        act_card = Card(frame, title="Activar licencia PRO")
        act_card.pack(fill="x", pady=10)

        act_frame = ctk.CTkFrame(act_card, fg_color="transparent")
        act_frame.pack(fill="x", padx=16, pady=8)

        self.license_entry = StyledEntry(act_frame, placeholder="Pega aquí tu clave de licencia...",
                                          width=500)
        self.license_entry.pack(side="left", padx=(0, 10))

        StyledButton(act_frame, text="✅ Activar", command=self._activate_license,
                     style="gold", width=120).pack(side="left")

        # Contacto
        contact_card = Card(frame, title="📞 Contacto para adquirir licencia")
        contact_card.pack(fill="x", pady=10)
        ctk.CTkLabel(contact_card,
                     text=f"📧 Email: {CONTACT_EMAIL}\n"
                     f"📱 Teléfono: {CONTACT_PHONE}\n\n"
                     "Para adquirir la versión PRO:\n"
                     "1. Copia el ID de este equipo\n"
                     "2. Envíalo por email junto con tus datos\n"
                     "3. Recibirás tu clave de activación personalizada",
                     font=AppStyles.FONT_BODY,
                     text_color=COLORS["text_light"],
                     justify="left").pack(padx=16, pady=(4, 16))

    def _activate_license(self):
        """Activa la licencia PRO."""
        key = self.license_entry.get().strip()
        if not key:
            messagebox.showwarning("Sin clave", "Introduce la clave de licencia.")
            return

        result = self.license_mgr.activate_license(key)
        if result["success"]:
            self.is_pro = True
            self.license_label.configure(text="🔓 PRO", text_color=COLORS["success"])
            messagebox.showinfo("¡Activada!",
                                f"Licencia PRO activada correctamente.\n"
                                f"Expira: {result.get('expires', 'N/A')}")
            self._show_welcome()
        else:
            messagebox.showerror("Error", result.get("error", "Clave inválida"))

    # ============================================================
    # PANTALLA: ACERCA DE
    # ============================================================

    def _show_about(self):
        """Pantalla Acerca de."""
        self._clear_main()
        self._highlight_nav("acerca")
        self.current_screen = "acerca"

        frame = ctk.CTkFrame(self.main_area, fg_color="transparent")
        frame.pack(fill="both", expand=True, padx=40, pady=30)

        # Logo
        if self.logo_image:
            ctk.CTkLabel(frame, image=self.logo_image, text="").pack(pady=(20, 10))

        ctk.CTkLabel(frame, text=APP_NAME,
                     font=("Segoe UI", 32, "bold"),
                     text_color=COLORS["gold"]).pack(pady=(10, 0))

        ctk.CTkLabel(frame, text=APP_SUBTITLE,
                     font=AppStyles.FONT_SUBTITLE,
                     text_color=COLORS["text_light"]).pack(pady=(4, 20))

        ctk.CTkLabel(frame, text=f"Versión {APP_VERSION} — © {YEAR}",
                     font=AppStyles.FONT_BODY,
                     text_color=COLORS["text_muted"]).pack(pady=2)

        # Autores
        authors_card = Card(frame, title="👥 Creadores")
        authors_card.pack(fill="x", pady=15)

        ctk.CTkLabel(authors_card, text=f"👨‍💻 {AUTHOR_1}",
                     font=("Segoe UI", 15, "bold"),
                     text_color=COLORS["text_white"]).pack(padx=16, pady=(8, 0), anchor="w")
        ctk.CTkLabel(authors_card, text=f"    {AUTHOR_1_TITLE}",
                     font=AppStyles.FONT_BODY,
                     text_color=COLORS["text_light"]).pack(padx=16, anchor="w")

        ctk.CTkLabel(authors_card, text=f"\n👩‍🍳 {AUTHOR_2}",
                     font=("Segoe UI", 15, "bold"),
                     text_color=COLORS["text_white"]).pack(padx=16, anchor="w")
        ctk.CTkLabel(authors_card, text=f"    {AUTHOR_2_TITLE}",
                     font=AppStyles.FONT_BODY,
                     text_color=COLORS["text_light"]).pack(padx=16, anchor="w", pady=(0, 12))

        # Contacto
        contact_card = Card(frame, title="📞 Contacto")
        contact_card.pack(fill="x", pady=10)
        ctk.CTkLabel(contact_card,
                     text=f"📧 {CONTACT_EMAIL}\n📱 {CONTACT_PHONE}",
                     font=("Segoe UI", 14),
                     text_color=COLORS["text_light"]).pack(padx=16, pady=12)

        # Tech stack
        tech_card = Card(frame, title="⚙️ Tecnología")
        tech_card.pack(fill="x", pady=10)
        ctk.CTkLabel(tech_card,
                     text="Python 3.11+ | CustomTkinter | pdfplumber | Matplotlib | Cryptography\n"
                     "Motor de análisis financiero propio basado en teoría de hostelería",
                     font=AppStyles.FONT_BODY,
                     text_color=COLORS["text_light"]).pack(padx=16, pady=12)


# ============================================================
# ENTRY POINT
# ============================================================

def main():
    app = RestaurantAIApp()
    app.mainloop()


if __name__ == "__main__":
    main()
