"""
Módulo de actualizaciones remotas — Convastro
La app consulta al servidor si hay una versión nueva disponible.
Si la hay, descarga el paquete y lo instala automáticamente.
"""

import os
import sys
import json
import shutil
import zipfile
import tempfile
from typing import Dict, Optional

try:
    import urllib.request
    import urllib.error
    HAS_URLLIB = True
except ImportError:
    HAS_URLLIB = False

# URL del servidor de actualizaciones
UPDATE_SERVER = "https://api.convastro.es"
CURRENT_VERSION = "1.1.0"


class AutoUpdater:
    """Gestiona actualizaciones remotas de la app."""

    def __init__(self, app_dir: str = None, version: str = None):
        self.app_dir = app_dir or os.path.dirname(os.path.abspath(__file__))
        self.version = version or CURRENT_VERSION
        self.update_info = None

    def check_for_updates(self) -> Dict:
        """Consulta al servidor si hay actualización disponible."""
        if not HAS_URLLIB:
            return {"update_available": False, "error": "urllib no disponible"}

        try:
            url = f"{UPDATE_SERVER}/api/updates/check?v={self.version}"
            req = urllib.request.Request(url, headers={"User-Agent": f"Convastro/{self.version}"})
            with urllib.request.urlopen(req, timeout=10) as resp:
                data = json.loads(resp.read().decode())
                self.update_info = data
                return data
        except urllib.error.URLError:
            return {"update_available": False, "error": "Sin conexión al servidor"}
        except Exception as e:
            return {"update_available": False, "error": str(e)}

    def download_update(self, download_url: str = None) -> Optional[str]:
        """Descarga el paquete de actualización. Devuelve path al zip."""
        if not self.update_info and not download_url:
            return None

        url = download_url or self.update_info.get("download_url", "")
        if not url:
            return None

        try:
            tmp_dir = tempfile.mkdtemp(prefix="convastro_update_")
            zip_path = os.path.join(tmp_dir, "update.zip")

            urllib.request.urlretrieve(url, zip_path)
            return zip_path
        except Exception as e:
            print(f"Error descargando actualización: {e}")
            return None

    def apply_update(self, zip_path: str) -> bool:
        """Aplica la actualización extrayendo el zip sobre la app actual."""
        try:
            # Backup
            backup_dir = self.app_dir + "_backup"
            if os.path.exists(backup_dir):
                shutil.rmtree(backup_dir)

            # Extraer
            tmp_extract = tempfile.mkdtemp(prefix="convastro_extract_")
            with zipfile.ZipFile(zip_path, 'r') as zf:
                zf.extractall(tmp_extract)

            # Buscar carpeta raíz dentro del zip
            contents = os.listdir(tmp_extract)
            if len(contents) == 1 and os.path.isdir(os.path.join(tmp_extract, contents[0])):
                source = os.path.join(tmp_extract, contents[0])
            else:
                source = tmp_extract

            # Copiar archivos actualizados (no borrar datos del usuario)
            for item in os.listdir(source):
                src = os.path.join(source, item)
                dst = os.path.join(self.app_dir, item)
                if os.path.isdir(src):
                    if os.path.exists(dst):
                        shutil.rmtree(dst)
                    shutil.copytree(src, dst)
                else:
                    shutil.copy2(src, dst)

            # Limpiar
            shutil.rmtree(tmp_extract, ignore_errors=True)
            os.remove(zip_path)

            return True
        except Exception as e:
            print(f"Error aplicando actualización: {e}")
            return False

    def get_changelog(self) -> str:
        """Devuelve el changelog de la actualización."""
        if self.update_info:
            return self.update_info.get("changelog", "Sin notas de versión.")
        return ""

    def get_update_summary(self) -> Dict:
        """Resumen legible de la actualización disponible."""
        if not self.update_info or not self.update_info.get("update_available"):
            return {"available": False, "message": f"Versión actual: {self.version} — al día"}

        return {
            "available": True,
            "current": self.version,
            "new_version": self.update_info.get("version", ""),
            "size_mb": self.update_info.get("size_mb", "?"),
            "changelog": self.update_info.get("changelog", ""),
            "message": f"Nueva versión {self.update_info.get('version','')} disponible",
        }
