"""
Módulo de persistencia — Guardar y cargar sesiones de análisis.
"""

import json
import os
from datetime import datetime
from typing import Dict, Optional


def save_session(datos, filepath: str) -> bool:
    """Guarda todos los datos del restaurante en un archivo JSON."""
    try:
        session = {
            "version": "1.1.0",
            "saved_at": datetime.now().isoformat(),
            "restaurante": {
                "nombre": datos.nombre_restaurante,
                "tipo": datos.tipo_restaurante,
                "clientes_mes": datos.clientes_mes,
                "dias_apertura_mes": datos.dias_apertura_mes,
                "servicios_por_dia": datos.servicios_por_dia,
                "personal_total": datos.personal_total,
                "asientos": datos.asientos,
            },
            "costes_fijos": {
                "alquiler": datos.alquiler,
                "salarios_fijos": datos.salarios_fijos,
                "seguridad_social": datos.seguridad_social,
                "seguros": datos.seguros,
                "suministros_fijos": datos.suministros_fijos,
                "amortizaciones": datos.amortizaciones,
                "otros_fijos": datos.otros_fijos,
            },
            "costes_variables": {
                "coste_mp_total": datos.coste_mp_total,
                "salarios_variables": datos.salarios_variables,
                "suministros_variables": datos.suministros_variables,
                "otros_variables": datos.otros_variables,
            },
            "stock": {
                "stock_inicial": datos.stock_inicial,
                "stock_final": datos.stock_final,
                "compras_periodo": datos.compras_periodo,
            },
            "platos": [
                {
                    "nombre": p.nombre,
                    "precio_venta": p.precio_venta,
                    "coste_mp": p.coste_mp,
                    "unidades_vendidas": p.unidades_vendidas,
                    "categoria": p.categoria,
                }
                for p in datos.platos
            ],
        }

        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(session, f, indent=2, ensure_ascii=False)
        return True
    except Exception as e:
        print(f"Error guardando sesión: {e}")
        return False


def load_session(filepath: str) -> Optional[Dict]:
    """Carga una sesión guardada."""
    try:
        with open(filepath, 'r', encoding='utf-8') as f:
            return json.load(f)
    except Exception as e:
        print(f"Error cargando sesión: {e}")
        return None


def apply_session(datos, session: Dict, Plato):
    """Aplica los datos de una sesión cargada al objeto DatosRestaurante."""
    r = session.get("restaurante", {})
    datos.nombre_restaurante = r.get("nombre", "")
    datos.tipo_restaurante = r.get("tipo", "tradicional")
    datos.clientes_mes = r.get("clientes_mes", 0)
    datos.dias_apertura_mes = r.get("dias_apertura_mes", 26)
    datos.servicios_por_dia = r.get("servicios_por_dia", 2)
    datos.personal_total = r.get("personal_total", 0)
    datos.asientos = r.get("asientos", 0)

    cf = session.get("costes_fijos", {})
    datos.alquiler = cf.get("alquiler", 0)
    datos.salarios_fijos = cf.get("salarios_fijos", 0)
    datos.seguridad_social = cf.get("seguridad_social", 0)
    datos.seguros = cf.get("seguros", 0)
    datos.suministros_fijos = cf.get("suministros_fijos", 0)
    datos.amortizaciones = cf.get("amortizaciones", 0)
    datos.otros_fijos = cf.get("otros_fijos", 0)

    cv = session.get("costes_variables", {})
    datos.coste_mp_total = cv.get("coste_mp_total", 0)
    datos.salarios_variables = cv.get("salarios_variables", 0)
    datos.suministros_variables = cv.get("suministros_variables", 0)
    datos.otros_variables = cv.get("otros_variables", 0)

    st = session.get("stock", {})
    datos.stock_inicial = st.get("stock_inicial", 0)
    datos.stock_final = st.get("stock_final", 0)
    datos.compras_periodo = st.get("compras_periodo", 0)

    datos.platos = []
    for p_data in session.get("platos", []):
        plato = Plato(
            nombre=p_data["nombre"],
            precio_venta=p_data["precio_venta"],
            coste_mp=p_data["coste_mp"],
            unidades_vendidas=p_data["unidades_vendidas"],
            categoria=p_data.get("categoria", "principal"),
        )
        datos.platos.append(plato)
