"""
Chatbot avanzado — RestaurantAI Pro
Puede explicar cómo se calculó cada métrica, dar consejos personalizados
y responder a preguntas complejas sobre el negocio.
"""

import json
import re
from datetime import datetime
from typing import Dict, List, Optional
from config import REF_MATERIA_PRIMA, REF_PERSONAL, REF_GASTOS_GENERALES, IVA_HOSTELERIA, BENCHMARKS


class RestaurantChatbot:
    """Chatbot avanzado con explicación de cálculos."""

    def __init__(self, datos_restaurante=None, analisis=None, ai_engine=None):
        self.datos = datos_restaurante
        self.analisis = analisis
        self.ai = ai_engine
        self.historial: List[Dict] = []

    def chat(self, mensaje: str) -> str:
        """Procesa mensaje y genera respuesta inteligente."""
        msg = mensaje.lower().strip()
        self.historial.append({"rol": "usuario", "msg": mensaje, "ts": datetime.now().isoformat()})

        # Detectar intención
        resp = None

        # EXPLICACIONES DE CÁLCULOS
        if any(w in msg for w in ["cómo calcula", "cómo se calcula", "explica", "fórmula",
                                   "por qué", "de dónde sale", "cómo has calculado"]):
            resp = self._explain_calculation(msg)

        # RESUMEN GENERAL
        elif any(w in msg for w in ["resumen", "mi restaurante", "cómo voy", "situación",
                                     "estado", "overview", "general"]):
            resp = self._resumen_general()

        # MENU ENGINEERING
        elif any(w in msg for w in ["menú", "carta", "plato", "estrella", "vaca",
                                     "puzzle", "perro", "engineering", "mejor plato",
                                     "peor plato", "qué elimino", "qué mantengo"]):
            resp = self._menu_engineering_chat(msg)

        # COSTES
        elif any(w in msg for w in ["coste", "gasto", "caro", "barato", "ahorro",
                                     "materia prima", "personal", "reducir", "ahorrar"]):
            resp = self._costes_chat(msg)

        # PRECIOS
        elif any(w in msg for w in ["precio", "subir", "bajar", "cobrar", "iva",
                                     "margen", "pvp", "cuánto cobro"]):
            resp = self._precios_chat(msg)

        # PUNTO MUERTO
        elif any(w in msg for w in ["punto muerto", "umbral", "break even", "equilibrio",
                                     "mínimo para", "cuánto necesito vender"]):
            resp = self._punto_muerto_chat()

        # STOCK
        elif any(w in msg for w in ["stock", "inventario", "almacén", "pedido",
                                     "proveedor", "compra", "wilson", "rotación"]):
            resp = self._stock_chat()

        # CLIENTES
        elif any(w in msg for w in ["cliente", "ocupación", "servicio", "ticket",
                                     "fidelizar", "captar", "comensales", "atraer"]):
            resp = self._clientes_chat()

        # PREDICCIONES IA
        elif any(w in msg for w in ["predic", "futuro", "pronóstico", "forecast",
                                     "siguiente mes", "próximo", "tendencia"]):
            resp = self._predicciones_chat()

        # BENCHMARKING
        elif any(w in msg for w in ["benchmark", "sector", "media", "comparar",
                                     "competencia", "referencia", "normal"]):
            resp = self._benchmark_chat()

        # WHAT-IF
        elif any(w in msg for w in ["qué pasa si", "what if", "simula", "escenario",
                                     "si subo", "si bajo", "si reduzco", "si aumento"]):
            resp = self._whatif_chat(msg)

        # LIQUIDEZ / CASH FLOW
        elif any(w in msg for w in ["liquidez", "cash flow", "tesorería", "pagar",
                                     "deuda", "flujo", "dinero"]):
            resp = self._liquidez_chat()

        # CONSEJOS
        elif any(w in msg for w in ["consejo", "mejora", "recomienda", "sugerencia",
                                     "qué hago", "ayuda", "tip"]):
            resp = self._consejos_chat()

        # SALUDO
        elif any(w in msg for w in ["hola", "buenos", "buenas", "qué tal", "hey"]):
            resp = self._saludo()

        if not resp:
            resp = self._respuesta_generica(msg)

        self.historial.append({"rol": "asistente", "msg": resp, "ts": datetime.now().isoformat()})
        return resp

    # ============================================================
    # EXPLICACIÓN DE CÁLCULOS (la función que pediste)
    # ============================================================

    def _explain_calculation(self, msg: str) -> str:
        """Explica paso a paso cómo se calculó cada métrica."""
        if not self.datos or not self.analisis:
            return "Necesito que cargues los datos del restaurante primero para explicar los cálculos."

        costes = self.analisis.analisis_costes()

        if any(w in msg for w in ["margen", "beneficio"]):
            ing = costes["ingresos"]
            ct = costes["costes_totales"]
            bn = costes["beneficio_neto"]
            return (
                f"📐 CÓMO SE CALCULA EL MARGEN NETO:\n\n"
                f"1. Sumo todos los ingresos de los platos:\n"
                f"   Σ (precio_venta × unidades_vendidas) = {ing:,.2f}€\n\n"
                f"2. Sumo todos los costes:\n"
                f"   Costes fijos: {self.datos.costes_fijos_total:,.2f}€\n"
                f"   + Costes variables: {self.datos.costes_variables_total:,.2f}€\n"
                f"   = Total costes: {ct:,.2f}€\n\n"
                f"3. Beneficio neto = Ingresos - Costes totales\n"
                f"   {ing:,.2f} - {ct:,.2f} = {bn:,.2f}€\n\n"
                f"4. Margen neto (%) = (Beneficio / Ingresos) × 100\n"
                f"   ({bn:,.2f} / {ing:,.2f}) × 100 = {costes['margen_neto_pct']:.1f}%\n\n"
                f"💡 Referencia: un margen neto >10% es saludable en hostelería."
            )

        if any(w in msg for w in ["punto muerto", "break even", "umbral"]):
            pm = self.analisis.punto_muerto()
            if "error" in pm:
                return f"No se puede calcular: {pm['error']}"
            return (
                f"📐 CÓMO SE CALCULA EL PUNTO MUERTO:\n\n"
                f"Fórmula: PM = Costes Fijos / Margen de Contribución\n\n"
                f"1. Costes fijos mensuales: {self.datos.costes_fijos_total:,.2f}€\n"
                f"   (alquiler + salarios fijos + seg.social + seguros + suministros + amortizaciones)\n\n"
                f"2. Precio medio por plato: {pm['precio_medio']:.2f}€\n"
                f"   (ingresos totales / platos vendidos totales)\n\n"
                f"3. Coste variable medio: {pm['coste_variable_medio']:.2f}€\n"
                f"   (costes variables totales / platos vendidos)\n\n"
                f"4. Margen de contribución = Precio medio - CV medio\n"
                f"   {pm['precio_medio']:.2f} - {pm['coste_variable_medio']:.2f} = {pm['margen_contribucion']:.2f}€\n\n"
                f"5. PM = {self.datos.costes_fijos_total:,.2f} / {pm['margen_contribucion']:.2f} = {pm['punto_muerto_unidades']:.0f} platos\n"
                f"   PM en euros = {pm['punto_muerto_unidades']:.0f} × {pm['precio_medio']:.2f} = {pm['punto_muerto_euros']:,.0f}€\n\n"
                f"📊 Vendes {pm['platos_actuales']} platos → {'✅ SUPERADO' if pm['superado'] else '❌ NO ALCANZADO'}\n"
                f"   Margen de seguridad: {pm['margen_seguridad_pct']:.1f}%"
            )

        if any(w in msg for w in ["menu engineering", "estrella", "popularidad", "índice"]):
            me = self.analisis.menu_engineering()
            if "error" in me:
                return f"No se puede calcular: {me['error']}"
            return (
                f"📐 CÓMO FUNCIONA EL MENU ENGINEERING:\n\n"
                f"Se clasifica cada plato según 2 índices:\n\n"
                f"1. Índice de Popularidad (IP):\n"
                f"   IP = (ventas del plato / ventas totales) × 100\n"
                f"   Umbral = (1 / nº platos) × 70% = {me['ip_umbral']:.1f}%\n"
                f"   Si IP ≥ umbral → POPULAR\n\n"
                f"2. Índice de Margen (IM):\n"
                f"   Margen medio ponderado = {me['margen_medio']:.2f}€\n"
                f"   IM = margen del plato / margen medio\n"
                f"   Si IM ≥ 1.0 → RENTABLE\n\n"
                f"Clasificación:\n"
                f"   Popular + Rentable = ⭐ ESTRELLA → mantener\n"
                f"   Popular + No rentable = 🐄 VACA → subir precio\n"
                f"   No popular + Rentable = 🧩 PUZZLE → promocionar\n"
                f"   No popular + No rentable = 🐕 PERRO → eliminar\n\n"
                f"Ejemplo: Si un plato vende 150 uds de 980 totales:\n"
                f"   IP = (150/980)×100 = 15.3% → ¿es >{me['ip_umbral']:.1f}%? {'Sí' if 15.3 > me['ip_umbral'] else 'No'}"
            )

        if any(w in msg for w in ["materia prima", "coste mp", "porcentaje mp"]):
            pct = costes["pct_materia_prima"]
            return (
                f"📐 CÓMO SE CALCULA EL % DE MATERIA PRIMA:\n\n"
                f"Fórmula: %CMP = (Coste MP total / Ingresos) × 100\n\n"
                f"Coste MP total: {self.datos.coste_mp_total:,.2f}€\n"
                f"Ingresos totales: {costes['ingresos']:,.2f}€\n\n"
                f"%CMP = ({self.datos.coste_mp_total:,.2f} / {costes['ingresos']:,.2f}) × 100 = {pct:.1f}%\n\n"
                f"Referencia sector: {REF_MATERIA_PRIMA*100:.0f}%\n"
                f"Tu resultado: {pct:.1f}% {'✅ Dentro de rango' if pct <= 42 else '⚠️ Por encima'}\n\n"
                f"Por plato individual:\n%CMP plato = (coste MP plato / precio venta) × 100"
            )

        if any(w in msg for w in ["ticket", "ticket medio"]):
            tm = self.datos.ticket_medio
            return (
                f"📐 CÓMO SE CALCULA EL TICKET MEDIO:\n\n"
                f"Fórmula: Ticket medio = Ingresos totales / Nº clientes\n\n"
                f"Ingresos: {costes['ingresos']:,.2f}€\n"
                f"Clientes/mes: {self.datos.clientes_mes}\n\n"
                f"Ticket medio = {costes['ingresos']:,.2f} / {self.datos.clientes_mes} = {tm:.2f}€\n\n"
                f"💡 Para subir el ticket medio:\n"
                f"- Upselling (sugerir extras, postres, vinos)\n"
                f"- Menús degustación\n"
                f"- Maridajes"
            )

        # Genérico
        return (
            "Puedo explicarte cómo se calculan estas métricas:\n\n"
            "• Margen neto / beneficio\n"
            "• Punto muerto / break even\n"
            "• Menu Engineering (estrellas, vacas, puzzles, perros)\n"
            "• % de materia prima\n"
            "• Ticket medio\n"
            "• Rotación de stock\n"
            "• Cash flow\n\n"
            "Pregúntame por cualquiera de ellas. Ejemplo:\n"
            "\"¿Cómo se calcula el punto muerto?\""
        )

    # ============================================================
    # RESPUESTAS CONTEXTUALES
    # ============================================================

    def _resumen_general(self) -> str:
        if not self.datos or not self.analisis:
            return "Carga los datos de tu restaurante para ver el resumen."
        costes = self.analisis.analisis_costes()
        cp = self.analisis.analisis_corto_plazo()
        return (
            f"📊 RESUMEN DE {self.datos.nombre_restaurante.upper()}\n\n"
            f"💰 Ingresos/mes: {costes['ingresos']:,.2f}€\n"
            f"📉 Costes/mes: {costes['costes_totales']:,.2f}€\n"
            f"💵 Beneficio: {costes['beneficio_neto']:,.2f}€\n"
            f"📊 Margen neto: {costes['margen_neto_pct']:.1f}%\n"
            f"🎫 Ticket medio: {cp['ticket_medio']:.2f}€\n"
            f"👥 Clientes/servicio: {cp['clientes_por_servicio']:.1f}\n"
            f"🏠 Ocupación: {cp['ocupacion_media']:.1f}%\n"
            f"💸 Cash flow: {cp['cash_flow_mensual']:,.2f}€\n"
            f"📋 Platos en carta: {len(self.datos.platos)}\n\n"
            + ("\n".join(costes.get("alertas", [])) if costes.get("alertas") else "✅ Sin alertas críticas")
        )

    def _menu_engineering_chat(self, msg: str) -> str:
        if not self.analisis:
            return "Carga datos y lanza el análisis primero."
        me = self.analisis.menu_engineering()
        if "error" in me:
            return f"No se puede analizar la carta: {me['error']}"

        if any(w in msg for w in ["mejor", "estrella", "mantengo"]):
            estrellas = [c for c in me["clasificacion"] if c["categoria"] == "estrella"]
            if estrellas:
                lines = [f"⭐ Tus ESTRELLAS (mantener y promocionar):\n"]
                for e in estrellas:
                    lines.append(f"  • {e['plato']}: PV {e['precio']:.2f}€, Margen {e['margen_bruto']:.2f}€, {e['unidades']} uds/mes")
                return "\n".join(lines)
            return "No tienes platos estrella actualmente. Revisa tu carta."

        if any(w in msg for w in ["peor", "perro", "elimino", "quito"]):
            perros = [c for c in me["clasificacion"] if c["categoria"] == "perro"]
            if perros:
                lines = ["🐕 Tus PERROS (considerar eliminar):\n"]
                for p in perros:
                    lines.append(f"  • {p['plato']}: PV {p['precio']:.2f}€, Margen {p['margen_bruto']:.2f}€, solo {p['unidades']} uds/mes")
                lines.append("\n💡 Elimina estos platos o reformula completamente la receta y precio.")
                return "\n".join(lines)
            return "¡No tienes platos perro! Tu carta está bien equilibrada."

        # Resumen completo
        lines = [f"🍽️ MENU ENGINEERING ({len(me['clasificacion'])} platos):\n"]
        for cat_key in ["estrella", "vaca", "puzzle", "perro"]:
            items = [c for c in me["clasificacion"] if c["categoria"] == cat_key]
            if items:
                from config import MENU_CATEGORIES
                info = MENU_CATEGORIES[cat_key]
                lines.append(f"\n{info['label']} ({len(items)}):")
                for i in items:
                    lines.append(f"  • {i['plato']} — PV:{i['precio']:.2f}€ Margen:{i['margen_bruto']:.2f}€ Uds:{i['unidades']}")
        lines.append("\n" + "\n".join(me.get("recomendaciones", [])[:3]))
        return "\n".join(lines)

    def _costes_chat(self, msg: str) -> str:
        if not self.analisis:
            return "Carga datos primero."
        costes = self.analisis.analisis_costes()

        if any(w in msg for w in ["reducir", "ahorrar", "bajar"]):
            return (
                "💡 ESTRATEGIAS PARA REDUCIR COSTES:\n\n"
                f"Tu % MP actual: {costes['pct_materia_prima']:.1f}% (ref: {REF_MATERIA_PRIMA*100:.0f}%)\n"
                f"Tu % Personal: {costes['pct_personal']:.1f}% (ref: {REF_PERSONAL*100:.0f}%)\n\n"
                "Materia prima:\n"
                "  1. Negociar volumen con proveedores\n"
                "  2. Comprar de temporada\n"
                "  3. Reducir mermas (FIFO estricto)\n"
                "  4. Estandarizar porciones (fichas técnicas)\n\n"
                "Personal:\n"
                "  1. Optimizar turnos según ocupación real\n"
                "  2. Formación para polivalencia\n"
                "  3. Incentivos por productividad\n\n"
                "Generales:\n"
                "  1. Renegociar alquiler o buscar alternativas\n"
                "  2. Comparar proveedores de suministros\n"
                "  3. Inversión en eficiencia energética"
            )

        return (
            f"💰 ESTRUCTURA DE COSTES:\n\n"
            f"Fijos: {self.datos.costes_fijos_total:,.2f}€ ({costes['pct_costes_fijos']:.1f}%)\n"
            f"Variables: {self.datos.costes_variables_total:,.2f}€ ({costes['pct_costes_variables']:.1f}%)\n"
            f"TOTAL: {costes['costes_totales']:,.2f}€ ({costes['pct_costes_totales']:.1f}%)\n\n"
            f"% MP: {costes['pct_materia_prima']:.1f}% {'✅' if costes['ref_mp_ok'] else '⚠️ ALTO'}\n"
            f"% Personal: {costes['pct_personal']:.1f}% {'✅' if costes['ref_personal_ok'] else '⚠️ ALTO'}\n\n"
            + ("\n".join(costes.get("alertas", [])))
        )

    def _precios_chat(self, msg: str) -> str:
        if not self.datos or not self.datos.platos:
            return "Añade platos a tu carta para análisis de precios."
        lines = ["💶 ANÁLISIS DE PRECIOS:\n"]
        for p in self.datos.platos:
            pv_sugerido = p.coste_mp / REF_MATERIA_PRIMA
            diff = ((p.precio_venta - pv_sugerido) / pv_sugerido * 100)
            estado = "✅" if abs(diff) < 15 else ("⬆️ subir" if diff < -15 else "⬇️ revisar")
            lines.append(f"  {p.nombre}: PV {p.precio_venta:.2f}€ | Sugerido {pv_sugerido:.2f}€ | {estado}")
        lines.append(f"\nPVP con IVA (10%) = PV × 1.10")
        return "\n".join(lines)

    def _punto_muerto_chat(self) -> str:
        if not self.analisis:
            return "Carga datos primero."
        pm = self.analisis.punto_muerto()
        if "error" in pm:
            return f"No se puede calcular: {pm['error']}"
        return (
            f"📈 PUNTO MUERTO:\n\n"
            f"Necesitas vender {pm['punto_muerto_unidades']:.0f} platos/mes ({pm['punto_muerto_euros']:,.0f}€)\n"
            f"Actualmente vendes: {pm['platos_actuales']} platos\n"
            f"Estado: {'✅ SUPERADO' if pm['superado'] else '❌ NO ALCANZADO'}\n"
            f"Margen de seguridad: {pm['margen_seguridad_pct']:.1f}%\n\n"
            f"💡 Para bajar el punto muerto: reduce costes fijos o aumenta margen por plato."
        )

    def _stock_chat(self) -> str:
        if not self.analisis:
            return "Carga datos primero."
        st = self.analisis.analisis_stock()
        return (
            f"📦 GESTIÓN DE STOCK:\n\n"
            f"Stock medio: {st['stock_medio']:,.2f}€\n"
            f"Consumo: {st['consumo_periodo']:,.2f}€\n"
            f"Rotación: {st['rotacion']:.1f} veces/año\n"
            f"Días de stock: {st['dias_stock']:.0f} días\n"
            f"Eficiencia: {st['eficiencia']}\n\n"
            f"💡 FIFO siempre. Ideal: rotación 20-50 para perecederos."
        )

    def _clientes_chat(self) -> str:
        if not self.analisis:
            return "Carga datos primero."
        cp = self.analisis.analisis_corto_plazo()
        return (
            f"👥 ANÁLISIS DE CLIENTES:\n\n"
            f"Clientes/mes: {self.datos.clientes_mes}\n"
            f"Por servicio: {cp['clientes_por_servicio']:.1f}\n"
            f"Ticket medio: {cp['ticket_medio']:.2f}€\n"
            f"Ocupación: {cp['ocupacion_media']:.1f}%\n\n"
            f"💡 Para aumentar clientes:\n"
            f"  1. Menú del día competitivo para mediodía\n"
            f"  2. Eventos temáticos\n"
            f"  3. Programa de fidelización\n"
            f"  4. Presencia en redes y Google Maps\n"
            f"  5. Alianzas con empresas cercanas"
        )

    def _predicciones_chat(self) -> str:
        if self.ai and self.ai.is_trained:
            preds = self.ai.predict_revenue(3)
            if preds:
                lines = ["🔮 PREDICCIÓN IA (próximos 3 meses):\n"]
                for p in preds:
                    lines.append(f"  {p['mes']}: {p['ingresos_predichos']:,.0f}€ "
                                 f"({p['clientes_estimados']} clientes, {p['confianza']}% confianza)")
                lines.append("\n📊 Modelo: Gradient Boosting con datos estacionales")
                return "\n".join(lines)
        return ("🔮 Las predicciones IA están disponibles en la versión PRO.\n"
                "Incluyen forecasting de ingresos y demanda basado en Machine Learning.")

    def _benchmark_chat(self) -> str:
        if not self.datos or not self.analisis:
            return "Carga datos primero."
        tipo = self.datos.tipo_restaurante.lower()
        bench = BENCHMARKS.get(tipo, BENCHMARKS.get("tradicional"))
        costes = self.analisis.analisis_costes()
        return (
            f"📊 BENCHMARKING vs SECTOR ({tipo.upper()}):\n\n"
            f"{'Métrica':<20} {'Tu dato':<12} {'Sector':<12} {'Estado'}\n"
            f"{'—'*56}\n"
            f"{'% Mat. prima':<20} {costes['pct_materia_prima']:<12.1f} {bench['pct_mp']:<12} {'✅' if costes['pct_materia_prima'] <= bench['pct_mp']*1.1 else '⚠️'}\n"
            f"{'% Personal':<20} {costes['pct_personal']:<12.1f} {bench['pct_personal']:<12} {'✅' if costes['pct_personal'] <= bench['pct_personal']*1.1 else '⚠️'}\n"
            f"{'Margen neto':<20} {costes['margen_neto_pct']:<12.1f} {bench['margen_neto']:<12} {'✅' if costes['margen_neto_pct'] >= bench['margen_neto']*0.8 else '⚠️'}\n"
            f"{'Ticket medio':<20} {self.datos.ticket_medio:<12.2f} {bench['ticket_medio']:<12} {'✅' if self.datos.ticket_medio >= bench['ticket_medio']*0.8 else '⚠️'}\n"
        )

    def _whatif_chat(self, msg: str) -> str:
        if not self.ai:
            return "La simulación what-if está disponible en la versión PRO."
        # Extraer porcentajes del mensaje
        nums = re.findall(r'(\d+)', msg)
        pct = int(nums[0]) if nums else 10

        if any(w in msg for w in ["subo", "subir", "precio"]):
            r = self.ai.whatif_simulation({"precio_cambio_pct": pct})
        elif any(w in msg for w in ["bajo", "bajar", "reduzco", "reducir", "coste"]):
            r = self.ai.whatif_simulation({"costes_mp_cambio_pct": -pct})
        elif any(w in msg for w in ["cliente", "aumento"]):
            r = self.ai.whatif_simulation({"clientes_cambio_pct": pct})
        else:
            r = self.ai.whatif_simulation({"precio_cambio_pct": 5, "clientes_cambio_pct": 10})

        return (
            f"🔮 SIMULACIÓN WHAT-IF:\n\n"
            f"{'Métrica':<20} {'Antes':<15} {'Después':<15}\n"
            f"{'—'*50}\n"
            f"{'Ingresos':<20} {r['ingresos_antes']:>12,.0f}€  {r['ingresos_despues']:>12,.0f}€\n"
            f"{'Costes':<20} {r['costes_antes']:>12,.0f}€  {r['costes_despues']:>12,.0f}€\n"
            f"{'Beneficio':<20} {r['beneficio_antes']:>12,.0f}€  {r['beneficio_despues']:>12,.0f}€\n"
            f"{'Margen':<20} {r['margen_antes']:>11.1f}%   {r['margen_despues']:>11.1f}%\n\n"
            f"Impacto: {'+' if r['impacto']>0 else ''}{r['impacto']:,.0f}€/mes"
        )

    def _liquidez_chat(self) -> str:
        if not self.analisis:
            return "Carga datos primero."
        cp = self.analisis.analisis_corto_plazo()
        return (
            f"💸 LIQUIDEZ Y CASH FLOW:\n\n"
            f"Cash flow mensual: {cp['cash_flow_mensual']:,.2f}€\n"
            f"Ingreso diario: {cp['ingreso_diario']:,.2f}€\n"
            f"Coste diario: {cp['coste_diario']:,.2f}€\n"
            f"{'✅ Positivo' if cp['liquidez_ok'] else '🚨 NEGATIVO — Riesgo de impago'}\n\n"
            f"💡 Consejos:\n"
            f"  1. Reserva 2-3 meses de costes fijos\n"
            f"  2. Negocia plazos con proveedores (30-60 días)\n"
            f"  3. Cobra al contado siempre que puedas\n"
            f"  4. No inmovilices dinero en stock excesivo"
        )

    def _consejos_chat(self) -> str:
        if not self.analisis:
            return "Carga datos para recibir consejos personalizados."
        costes = self.analisis.analisis_costes()
        me = self.analisis.menu_engineering()
        tips = ["💡 CONSEJOS PERSONALIZADOS:\n"]

        if costes["pct_materia_prima"] > 40:
            tips.append("🔴 Tu coste de MP es alto. Negocia con proveedores o ajusta porciones.")
        if costes["margen_neto_pct"] < 8:
            tips.append("🔴 Margen muy bajo. Necesitas subir precios o reducir costes ya.")
        if "resumen" in me:
            n_perros = me["resumen"].get("perro", {}).get("count", 0)
            if n_perros > 2:
                tips.append(f"🟡 Tienes {n_perros} platos 'perro'. Elimina al menos la mitad.")
            n_estrellas = me["resumen"].get("estrella", {}).get("count", 0)
            if n_estrellas > 0:
                tips.append(f"🟢 Tienes {n_estrellas} estrellas. Destácalas en la carta y promuévelas.")

        if len(tips) == 1:
            tips.append("✅ Tu restaurante está razonablemente bien. Sigue monitorizando.")
        return "\n".join(tips)

    def _saludo(self) -> str:
        nombre = self.datos.nombre_restaurante if self.datos else "tu restaurante"
        return (
            f"👋 ¡Hola! Soy el asistente de RestaurantAI Pro.\n\n"
            f"Puedo ayudarte con:\n"
            f"  📊 \"resumen\" — Estado general de {nombre}\n"
            f"  🍽️ \"carta\" — Menu Engineering\n"
            f"  💰 \"costes\" — Análisis de costes\n"
            f"  📈 \"punto muerto\" — Umbral de rentabilidad\n"
            f"  🔮 \"predicciones\" — Forecasting con IA\n"
            f"  📐 \"cómo se calcula el margen\" — Explicación de fórmulas\n"
            f"  📊 \"benchmark\" — Comparativa con el sector\n"
            f"  🔮 \"qué pasa si subo precios 10%\" — Simulaciones\n"
            f"  💡 \"consejos\" — Recomendaciones personalizadas\n"
        )

    def _respuesta_generica(self, msg: str) -> str:
        return (
            "No he entendido exactamente tu pregunta. Prueba con:\n"
            "  • \"resumen\" — Estado del restaurante\n"
            "  • \"carta\" — Menu Engineering\n"
            "  • \"cómo se calcula el margen\" — Explicaciones\n"
            "  • \"qué pasa si subo precios 10%\" — Simulación\n"
            "  • \"consejos\" — Recomendaciones\n"
            "  • \"benchmark\" — Comparar con el sector"
        )

    # ============================================================
    # SOPORTE PERIÓDICO
    # ============================================================

    def get_soporte_diario(self) -> str:
        if not self.analisis:
            return "Carga datos para el soporte diario."
        cp = self.analisis.analisis_corto_plazo()
        tips = [f"📅 BRIEFING DIARIO — {datetime.now().strftime('%d/%m/%Y')}\n"]
        tips.append(f"🎯 Objetivo de ventas hoy: {cp['ingreso_diario']:,.0f}€")
        tips.append(f"👥 Clientes esperados por servicio: {cp['clientes_por_servicio']:.0f}")
        if cp.get("ocupacion_media", 0) < 50:
            tips.append("💡 Ocupación baja: considera oferta especial para mediodía")
        if not cp.get("liquidez_ok"):
            tips.append("⚠️ Cash flow negativo. Prioriza cobros hoy.")
        tips.append(f"\n✅ Objetivo: superar {cp['ingreso_diario']:,.0f}€ en ventas")
        return "\n".join(tips)

    def get_soporte_semanal(self) -> str:
        if not self.analisis:
            return "Carga datos para el informe semanal."
        me = self.analisis.menu_engineering()
        costes = self.analisis.analisis_costes()
        lines = [f"📊 INFORME SEMANAL\n"]
        lines.append(f"Margen actual: {costes['margen_neto_pct']:.1f}%")
        if "resumen" in me:
            for cat, data in me["resumen"].items():
                from config import MENU_CATEGORIES
                lines.append(f"  {MENU_CATEGORIES[cat]['label']}: {data['count']} platos ({data['beneficio']:,.0f}€)")
        lines.append("\nAcciones esta semana:")
        for r in me.get("recomendaciones", [])[:2]:
            lines.append(f"  {r}")
        return "\n".join(lines)

    def get_soporte_mensual(self) -> str:
        if not self.analisis:
            return "Carga datos para el informe mensual."
        informe = self.analisis.generar_informe_completo(es_pro=True)
        c = informe["estructura_costes"]
        lines = [
            f"📈 INFORME MENSUAL COMPLETO",
            f"{'='*40}\n",
            f"💰 Ingresos: {c['ingresos']:,.2f}€",
            f"📉 Costes: {c['costes_totales']:,.2f}€",
            f"💵 Beneficio: {c['beneficio_neto']:,.2f}€",
            f"📊 Margen: {c['margen_neto_pct']:.1f}%",
            f"\n% MP: {c['pct_materia_prima']:.1f}% {'✅' if c['ref_mp_ok'] else '⚠️'}",
            f"% Personal: {c['pct_personal']:.1f}% {'✅' if c['ref_personal_ok'] else '⚠️'}",
        ]
        pm = informe.get("punto_muerto", {})
        if "punto_muerto_unidades" in pm:
            lines.append(f"\nPunto muerto: {pm['punto_muerto_unidades']:.0f} platos")
            lines.append(f"Margen seguridad: {pm['margen_seguridad_pct']:.1f}%")
        return "\n".join(lines)
