"""
Motor de Análisis Financiero para Restauración
Fórmulas: Costes, PV, Márgenes, Punto muerto, Menu Engineering, Stock, Wilson
"""

import math
from typing import List, Dict
from config import REF_MATERIA_PRIMA, REF_PERSONAL, REF_GASTOS_GENERALES, IVA_HOSTELERIA, MENU_CATEGORIES


class Plato:
    def __init__(self, nombre: str, precio_venta: float, coste_mp: float,
                 unidades_vendidas: int, categoria: str = "principal",
                 coste_mo_unitario: float = 0, costes_generales_unitario: float = 0):
        self.nombre = nombre
        self.precio_venta = precio_venta
        self.coste_mp = coste_mp
        self.unidades_vendidas = unidades_vendidas
        self.categoria = categoria
        self.coste_mo_unitario = coste_mo_unitario
        self.costes_generales_unitario = costes_generales_unitario

    @property
    def coste_total(self): return self.coste_mp + self.coste_mo_unitario + self.costes_generales_unitario
    @property
    def margen_bruto(self): return self.precio_venta - self.coste_mp
    @property
    def margen_neto(self): return self.precio_venta - self.coste_total
    @property
    def pct_coste_mp(self): return (self.coste_mp / self.precio_venta * 100) if self.precio_venta > 0 else 0
    @property
    def pct_margen(self): return (self.margen_neto / self.precio_venta * 100) if self.precio_venta > 0 else 0
    @property
    def pvp_con_iva(self): return self.precio_venta * (1 + IVA_HOSTELERIA)
    @property
    def ingresos_totales(self): return self.precio_venta * self.unidades_vendidas
    @property
    def beneficio_total(self): return self.margen_neto * self.unidades_vendidas


class DatosRestaurante:
    def __init__(self):
        self.nombre_restaurante: str = ""
        self.tipo_restaurante: str = "tradicional"
        self.platos: List[Plato] = []
        self.clientes_mes: int = 0
        self.dias_apertura_mes: int = 26
        self.servicios_por_dia: int = 2
        self.personal_total: int = 0
        self.asientos: int = 0
        self.alquiler: float = 0
        self.seguros: float = 0
        self.salarios_fijos: float = 0
        self.seguridad_social: float = 0
        self.suministros_fijos: float = 0
        self.amortizaciones: float = 0
        self.otros_fijos: float = 0
        self.coste_mp_total: float = 0
        self.salarios_variables: float = 0
        self.suministros_variables: float = 0
        self.otros_variables: float = 0
        self.stock_inicial: float = 0
        self.stock_final: float = 0
        self.compras_periodo: float = 0
        self.historico_mensual: List[Dict] = []

    @property
    def costes_fijos_total(self):
        return (self.alquiler + self.seguros + self.salarios_fijos +
                self.seguridad_social + self.suministros_fijos +
                self.amortizaciones + self.otros_fijos)
    @property
    def costes_variables_total(self):
        return (self.coste_mp_total + self.salarios_variables +
                self.suministros_variables + self.otros_variables)
    @property
    def costes_totales(self): return self.costes_fijos_total + self.costes_variables_total
    @property
    def ingresos_totales(self): return sum(p.ingresos_totales for p in self.platos)
    @property
    def ticket_medio(self):
        return self.ingresos_totales / self.clientes_mes if self.clientes_mes > 0 else 0
    @property
    def clientes_por_servicio(self):
        ts = self.dias_apertura_mes * self.servicios_por_dia
        return self.clientes_mes / ts if ts > 0 else 0
    @property
    def ocupacion_media(self):
        if self.asientos == 0: return 0
        return (self.clientes_por_servicio / self.asientos) * 100


class AnalisisFinanciero:
    def __init__(self, datos: DatosRestaurante):
        self.datos = datos

    def analisis_costes(self) -> Dict:
        ing = self.datos.ingresos_totales
        cf = self.datos.costes_fijos_total
        cv = self.datos.costes_variables_total
        ct = self.datos.costes_totales
        pct_cf = (cf / ing * 100) if ing > 0 else 0
        pct_cv = (cv / ing * 100) if ing > 0 else 0
        pct_ct = (ct / ing * 100) if ing > 0 else 0
        pct_mp = (self.datos.coste_mp_total / ing * 100) if ing > 0 else 0
        pct_per = ((self.datos.salarios_fijos + self.datos.seguridad_social +
                    self.datos.salarios_variables) / ing * 100) if ing > 0 else 0

        alertas = []
        if pct_mp > REF_MATERIA_PRIMA * 100 * 1.15:
            alertas.append(f"⚠️ Materia prima ({pct_mp:.1f}%) por encima del {REF_MATERIA_PRIMA*100:.0f}%.")
        if pct_per > REF_PERSONAL * 100 * 1.15:
            alertas.append(f"⚠️ Personal ({pct_per:.1f}%) elevado (ref: {REF_PERSONAL*100:.0f}%).")
        if pct_ct > 85:
            alertas.append("🚨 Costes >85% de ingresos. Margen muy ajustado.")
        if pct_ct > 95:
            alertas.append("🚨 CRÍTICO: costes casi igualan ingresos.")

        return {
            "ingresos": ing, "costes_fijos": cf, "costes_variables": cv,
            "costes_totales": ct, "beneficio_bruto": ing - cv,
            "beneficio_neto": ing - ct,
            "pct_costes_fijos": pct_cf, "pct_costes_variables": pct_cv,
            "pct_costes_totales": pct_ct,
            "pct_materia_prima": pct_mp, "pct_personal": pct_per,
            "ref_mp_ok": pct_mp <= REF_MATERIA_PRIMA * 100 * 1.1,
            "ref_personal_ok": pct_per <= REF_PERSONAL * 100 * 1.1,
            "margen_neto_pct": ((ing - ct) / ing * 100) if ing > 0 else 0,
            "alertas": alertas,
        }

    def punto_muerto(self) -> Dict:
        cf = self.datos.costes_fijos_total
        ing = self.datos.ingresos_totales
        cv = self.datos.costes_variables_total
        tp = sum(p.unidades_vendidas for p in self.datos.platos)
        if tp == 0: return {"error": "No hay datos de ventas"}
        pm = ing / tp if tp > 0 else 0
        cvm = cv / tp if tp > 0 else 0
        mc = pm - cvm
        if mc <= 0: return {"error": "Margen de contribución negativo", "margen_contribucion": mc}
        pmu = cf / mc
        pme = pmu * pm
        ms = ((tp - pmu) / tp * 100) if tp > 0 else 0
        return {
            "punto_muerto_unidades": pmu, "punto_muerto_euros": pme,
            "precio_medio": pm, "coste_variable_medio": cvm,
            "margen_contribucion": mc, "platos_actuales": tp,
            "margen_seguridad_pct": ms, "superado": tp > pmu,
            "platos_para_rentabilidad": max(0, math.ceil(pmu) - tp),
        }

    def menu_engineering(self) -> Dict:
        if not self.datos.platos: return {"error": "No hay platos"}
        tv = sum(p.unidades_vendidas for p in self.datos.platos)
        n = len(self.datos.platos)
        if tv == 0 or n == 0: return {"error": "No hay ventas"}
        mm = sum(p.margen_bruto * p.unidades_vendidas for p in self.datos.platos) / tv
        ipu = (1 / n) * 0.70 * 100

        clasificacion = []
        for p in self.datos.platos:
            ip = (p.unidades_vendidas / tv) * 100
            im = p.margen_bruto / mm if mm > 0 else 0
            popular = ip >= ipu
            rentable = im >= 1.0
            if popular and rentable: cat = "estrella"
            elif popular: cat = "vaca"
            elif rentable: cat = "puzzle"
            else: cat = "perro"
            clasificacion.append({
                "plato": p.nombre, "precio": p.precio_venta, "coste_mp": p.coste_mp,
                "margen_bruto": p.margen_bruto, "unidades": p.unidades_vendidas,
                "indice_popularidad": ip, "indice_margen": im,
                "categoria": cat, "categoria_info": MENU_CATEGORIES[cat],
                "ingresos_totales": p.ingresos_totales,
                "beneficio_total": p.margen_bruto * p.unidades_vendidas,
            })

        resumen = {}
        for ck in MENU_CATEGORIES:
            pc = [c for c in clasificacion if c["categoria"] == ck]
            resumen[ck] = {"count": len(pc),
                           "ingresos": sum(c["ingresos_totales"] for c in pc),
                           "beneficio": sum(c["beneficio_total"] for c in pc),
                           "platos": [c["plato"] for c in pc]}

        recs = []
        estrellas = [c for c in clasificacion if c["categoria"] == "estrella"]
        vacas = [c for c in clasificacion if c["categoria"] == "vaca"]
        puzzles = [c for c in clasificacion if c["categoria"] == "puzzle"]
        perros = [c for c in clasificacion if c["categoria"] == "perro"]

        if estrellas:
            recs.append(f"✅ ESTRELLAS ({len(estrellas)}): {', '.join(c['plato'] for c in estrellas[:3])}. Mantener y promocionar.")
        if vacas:
            for v in vacas[:3]:
                ps = v["coste_mp"] / REF_MATERIA_PRIMA
                recs.append(f"🔄 VACA '{v['plato']}': Margen bajo ({v['margen_bruto']:.2f}€). Considerar subir a ~{ps:.2f}€.")
        if puzzles:
            for p in puzzles[:3]:
                recs.append(f"📢 PUZZLE '{p['plato']}': Buen margen pero pocas ventas. Reposicionar en carta.")
        if perros:
            recs.append(f"❌ PERROS ({len(perros)}): {', '.join(c['plato'] for c in perros[:3])}. Evaluar eliminación.")

        return {"clasificacion": clasificacion, "resumen": resumen,
                "margen_medio": mm, "ip_umbral": ipu, "total_vendidos": tv,
                "recomendaciones": recs}

    def analisis_stock(self) -> Dict:
        si, sf = self.datos.stock_inicial, self.datos.stock_final
        sm = (si + sf) / 2 if (si + sf) > 0 else 0
        consumo = si + self.datos.compras_periodo - sf
        rot = consumo / sm if sm > 0 else 0
        dias = 365 / rot if rot > 0 else float('inf')
        return {"stock_medio": sm, "consumo_periodo": consumo, "rotacion": rot,
                "dias_stock": dias,
                "eficiencia": "Buena" if 20 <= rot <= 50 else "Baja" if rot < 20 else "Muy alta"}

    def modelo_wilson(self, D, S, H) -> Dict:
        if H <= 0: return {"error": "H debe ser > 0"}
        q = math.sqrt((2 * D * S) / H)
        n = D / q if q > 0 else 0
        ct = (n * S) + (q / 2 * H)
        return {"cantidad_optima": q, "numero_pedidos_año": n,
                "dias_entre_pedidos": 365 / n if n > 0 else 0, "coste_total_gestion": ct}

    def analisis_corto_plazo(self) -> Dict:
        ing = self.datos.ingresos_totales
        ct = self.datos.costes_totales
        d = self.datos.dias_apertura_mes or 1
        s = self.datos.servicios_por_dia or 1
        id = ing / d
        cd = ct / d
        iser = id / s
        cf = (ing - ct) + self.datos.amortizaciones
        alertas = []
        if cf < 0: alertas.append("🚨 Cash flow negativo.")
        if id < cd: alertas.append("⚠️ Costes diarios superan ingresos.")
        if self.datos.ocupacion_media < 40: alertas.append("⚠️ Ocupación baja (<40%).")
        return {
            "ingreso_diario": id, "coste_diario": cd, "beneficio_diario": id - cd,
            "ingreso_por_servicio": iser, "ticket_medio": self.datos.ticket_medio,
            "clientes_por_servicio": self.datos.clientes_por_servicio,
            "ocupacion_media": self.datos.ocupacion_media,
            "cash_flow_mensual": cf, "liquidez_ok": cf > 0,
            "stock": self.analisis_stock(), "alertas": alertas,
        }

    def analisis_medio_plazo(self) -> Dict:
        me = self.menu_engineering()
        pm = self.punto_muerto()
        costes = self.analisis_costes()
        ing = self.datos.ingresos_totales
        ct = self.datos.costes_totales
        sims = []
        # +5% precios
        ni = ing * 1.05
        sims.append({"nombre": "Subir precios 5%", "impacto_ingresos": ni - ing,
                      "nuevo_margen": ((ni - ct) / ni * 100) if ni > 0 else 0})
        # -10% MP
        ncv = self.datos.costes_variables_total * 0.90
        nct = self.datos.costes_fijos_total + ncv
        sims.append({"nombre": "Reducir coste MP 10%", "ahorro": self.datos.costes_variables_total - ncv,
                      "nuevo_margen": ((ing - nct) / ing * 100) if ing > 0 else 0})
        # +15% clientes
        ni3 = ing * 1.15
        ncv3 = self.datos.costes_variables_total * 1.15
        nct3 = self.datos.costes_fijos_total + ncv3
        sims.append({"nombre": "Aumentar clientes 15%", "impacto_ingresos": ni3 - ing,
                      "nuevo_margen": ((ni3 - nct3) / ni3 * 100) if ni3 > 0 else 0})

        recs = []
        if costes["pct_materia_prima"] > REF_MATERIA_PRIMA * 100:
            recs.append("📉 Negociar con proveedores para reducir MP.")
        if costes["margen_neto_pct"] < 10:
            recs.append("📊 Margen bajo. Priorizar estrellas y eliminar perros.")
        return {
            "menu_engineering": me, "punto_muerto": pm, "estructura_costes": costes,
            "ingresos_proyectados_trim": ing * 3, "beneficio_proyectado_trim": (ing - ct) * 3,
            "simulaciones": sims, "recomendaciones": recs,
        }

    def analisis_largo_plazo(self) -> Dict:
        ia = self.datos.ingresos_totales * 12
        ca = self.datos.costes_totales * 12
        ba = ia - ca
        inv = self.datos.costes_fijos_total * 24
        re = (ba / inv * 100) if inv > 0 else 0
        cfa = ba + (self.datos.amortizaciones * 12)
        roi = re
        pb = inv / (ba / 12) if ba > 0 else float('inf')

        proy = []
        for y in range(1, 4):
            fg = 1 + 0.03 * y
            fc = 1 + 0.02 * y
            i, c = ia * fg, ca * fc
            proy.append({"año": y, "ingresos": i, "costes": c, "beneficio": i - c,
                          "margen": ((i - c) / i * 100) if i > 0 else 0})

        if ba < 0: estado, color, msg = "crítico", "danger", "Genera pérdidas. Acción inmediata."
        elif re < 5: estado, color, msg = "débil", "warning", "Rentabilidad baja."
        elif re < 15: estado, color, msg = "estable", "gold", "Aceptable. Margen de mejora."
        else: estado, color, msg = "saludable", "success", "Buena rentabilidad."

        plan = []
        if ba < 0:
            plan = ["1. URGENTE: Revisar costes fijos", "2. Rediseñar carta", "3. Control stocks estricto", "4. Considerar cambio de modelo"]
        elif re < 10:
            plan = ["1. Menu Engineering trimestral", "2. Negociar proveedores", "3. Fidelización", "4. Inversión en eficiencia"]
        else:
            plan = ["1. Mantener calidad", "2. Considerar expansión", "3. Invertir en marca", "4. Diversificar ingresos"]

        return {
            "ingresos_anuales": ia, "costes_anuales": ca, "beneficio_anual": ba,
            "rentabilidad_economica": re, "cash_flow_anual": cfa, "roi": roi,
            "payback_meses": pb, "proyeccion_3_años": proy,
            "salud_financiera": {"estado": estado, "color": color, "mensaje": msg},
            "plan_accion": plan,
        }

    def generar_informe_completo(self, es_pro: bool = False) -> Dict:
        informe = {
            "restaurante": self.datos.nombre_restaurante, "tipo": self.datos.tipo_restaurante,
            "fecha": __import__('datetime').datetime.now().isoformat(),
            "corto_plazo": self.analisis_corto_plazo(),
            "menu_engineering": self.menu_engineering(),
            "punto_muerto": self.punto_muerto(),
            "estructura_costes": self.analisis_costes(),
        }
        if es_pro:
            informe["medio_plazo"] = self.analisis_medio_plazo()
            informe["largo_plazo"] = self.analisis_largo_plazo()
        return informe
